<?php

namespace Drupal\ai_automator_extractor\Plugin\AiAutomatorType;

use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The rules for an file field.
 */
#[AiAutomatorType(
  id: 'ai_automator_extractor_file',
  label: new TranslatableMarkup('Extractor: File'),
  field_rule: 'file',
  target: 'file',
)]
class FileExtractor extends ExternalBase implements AiAutomatorTypeInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Extractor: File';

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $values = [];
    $config = $fieldDefinition->getConfig($entity->bundle())->getSettings();
    $exts = str_replace(" ", "|", $config['file_extensions']);
    foreach ($entity->{$automatorConfig['base_field']} as $wrapperEntity) {
      preg_match_all("/(http[s]*:\/\/)([a-z\-_0-9\/.]+)\.([a-z.]{2,3})\/([a-z0-9\-_\/._~:?#\[\]@!$&'()*+,;=%]*)([a-z0-9]+\.)($exts)/i", $wrapperEntity->value, $matches);
      $values = [];
      if (isset($matches[0][0])) {
        foreach ($matches[0] as $mail) {
          $values[$mail] = $mail;
        }
      }
    }
    return array_values($values);
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $config = $fieldDefinition->getConfig($entity->bundle())->getSettings();
    if (!filter_var($value, FILTER_VALIDATE_URL)) {
      return FALSE;
    }
    // File is allowed.
    $allowedExtension = FALSE;
    foreach (explode(" ", $config['file_extensions']) as $ext) {
      if (substr($value, -(strlen($ext))) == $ext) {
        $allowedExtension = TRUE;
      }
    }

    if (!$allowedExtension) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $fileHelper = $this->getFileHelper();

    // Transform string to boolean.
    $fileEntities = [];

    // Successful counter, to only download as many as max.
    $successFul = 0;
    foreach ($values as $value) {
      // Everything validated, then we prepare the file path to save to.
      $filePath = $fileHelper->createFilePathFromFieldConfig(basename($value), $fieldDefinition, $entity);
      // Create file entity from string.
      $file = $fileHelper->generateFileFromBinary(file_get_contents($value), $filePath);
      // If we can save, we attach it.
      if ($file) {
        // Add to the entities list.
        $fileEntities[] = [
          'target_id' => $file->id(),
        ];

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality()) {
          break;
        }
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $fileEntities);
  }

}
