<?php

namespace Drupal\ai_automator_extractor\Plugin\AiAutomatorType;

use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The rules for an image field.
 */
#[AiAutomatorType(
  id: 'ai_automator_extractor_image',
  label: new TranslatableMarkup('Extractor: Image'),
  field_rule: 'image',
  target: 'file',
)]
class ImageExtractor extends ExternalBase implements AiAutomatorTypeInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Extractor: Image';

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function extraAdvancedFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, FormStateInterface $formState, array $defaultValues = []) {

    $form['automator_extractor_offset'] = [
      '#type' => 'number',
      '#title' => 'Offset',
      '#description' => $this->t('Offset to start from. If you want to start from the second one, you write 2 here.'),
      '#default_value' => $defaultValues['automator_extractor_offset'] ?? '',
      '#weight' => 24,
    ];

    $form['automator_extractor_amount'] = [
      '#type' => 'number',
      '#title' => 'Amount',
      '#description' => $this->t('Amount to get. If nothing is set it gets all if that is possible.'),
      '#default_value' => $defaultValues['automator_extractor_amount'] ?? '',
      '#weight' => 24,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $values = [];
    foreach ($entity->{$automatorConfig['base_field']} as $wrapperEntity) {
      preg_match_all("/(http[s]*:\/\/)([a-z\-_0-9\/.]+)\.([a-z.]{2,3})\/([a-z0-9\-_\/._~:?#\[\]@!$&'()*+,;=%]*)([a-z0-9]+\.)(gif|jpg|jpeg|png)/i", $wrapperEntity->value, $matches);
      $values = [];
      if (isset($matches[0][0])) {
        foreach ($matches[0] as $mail) {
          $values[$mail] = $mail;
        }
      }
    }
    return array_values($values);
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $config = $fieldDefinition->getConfig($entity->bundle())->getSettings();
    if (!filter_var($value, FILTER_VALIDATE_URL)) {
      return FALSE;
    }
    // File is allowed.
    $allowedExtension = FALSE;
    foreach (explode(" ", $config['file_extensions']) as $ext) {
      if (substr($value, -(strlen($ext))) == $ext) {
        $allowedExtension = TRUE;
      }
    }

    if (!$allowedExtension) {
      return FALSE;
    }
    return TRUE;
  }

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $config = $fieldDefinition->getConfig($entity->bundle())->getSettings();
    $fileHelper = $this->getFileHelper();
    // Transform string to boolean.
    $fileEntities = [];

    // Successful counter, to only download as many as max.
    $successFul = 0;
    $amount = $automatorConfig['extractor_amount'] ?? '';
    $offset = $automatorConfig['extractor_offset'] ?? '';
    $count = 0;
    foreach ($values as $value) {
      $count++;
      // Everything validated, then we prepare the file path to save to.
      $filePath = $fileHelper->createFilePathFromFieldConfig(basename($value), $fieldDefinition, $entity);
      // Get resolution.
      $resolution = getimagesize($value);
      // Check if resolution fits.
      if (!$this->fitsResolution($resolution, $config)) {
        continue;
      }
      if ($offset && $count < $offset) {
        continue;
      }
      // Create file entity from string.
      $image = $fileHelper->generateImageMetaDataFromBinary(file_get_contents($value), $filePath);
      // If we can save, we attach it.
      if ($image) {
        // Add to the entities list.
        $fileEntities[] = $image;

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality() || $successFul == $amount) {
          break;
        }
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $fileEntities);
  }

  /**
   * Check if resolution fits.
   *
   * @param array $resolution
   *   The resolution.
   * @param array $config
   *   The config.
   *
   * @return bool
   *   If it fits or not.
   */
  protected function fitsResolution($resolution, $config) {
    $max = explode('x', $config['max_resolution']);
    $min = explode('x', $config['min_resolution']);

    if (count($max) > 1 && ($resolution[0] >= $max[0] || $resolution[1] >= $max[1])) {
      return FALSE;
    }
    if (count($min) && ($resolution[0] <= $min[0] || $resolution[1] <= $min[1])) {
      return FALSE;
    }
    return TRUE;
  }

}
