<?php

namespace Drupal\ai_automator_extractor\Plugin\AiAutomatorType;

use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The rules for a RegEx field.
 */
#[AiAutomatorType(
  id: 'ai_automator_extractor_regex_counter',
  label: new TranslatableMarkup('Extractor: RegEx Counter'),
  field_rule: 'integer',
  target: '',
)]
class RegExCounter extends ExternalBase implements AiAutomatorTypeInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Extractor: RegEx Counter';

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function allowedInputs() {
    return [
      'text_long',
      'text',
      'string',
      'string_long',
      'text_with_summary',
      'link',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function extraAdvancedFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, FormStateInterface $formState, array $defaultValues = []) {
    $form['automator_extractor_regex'] = [
      '#type' => 'textfield',
      '#title' => 'Regular Expression',
      '#description' => $this->t('The regular expression to extract from. Try you rule on <a href="https://regex101.com/" target="blank">https://regex101.com/</a>.'),
      '#default_value' => $defaultValues['automator_extractor_regex'] ?? "",
      '#weight' => 24,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $fieldType = $entity->{$automatorConfig['base_field']}->getFieldDefinition()->getType();
    $values = [];
    foreach ($entity->{$automatorConfig['base_field']} as $wrapperEntity) {
      $matches = [];
      try {
        switch ($fieldType) {
          case 'link':
            $values[] = preg_match_all($automatorConfig['extractor_regex'], htmlspecialchars_decode($wrapperEntity->uri), $matches) ?? 0;
            break;
          default:
            $values[] = preg_match_all($automatorConfig['extractor_regex'], htmlspecialchars_decode($wrapperEntity->value), $matches) ?? 0;
        }
      }
      catch (\Exception $e) {
        $values[] = 0;
      }
    }
    return $values;
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    // Has to have a link an be valid.
    if (empty($value) || !is_numeric($value)) {
      return FALSE;
    }

    // Otherwise it is ok.
    return TRUE;
  }

}
