<?php

namespace Drupal\ai_automator_pixabay\Plugin\AiAutomatorType;

use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media_pixabay\Api\PixabayApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The rules for an image field.
 */
#[AiAutomatorType(
  id: 'ai_automator_pixabay_image',
  label: new TranslatableMarkup('Pixabay: Image Fetcher'),
  field_rule: 'image',
  target: 'file',
)]
class PixabayImage extends ExternalBase implements AiAutomatorTypeInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Pixabay Image Fetcher';

  /**
   * The media Pixabay API key config.
   */
  protected string $pixabayKeyConfig;

  /**
   * The pixabay api service.
   */
  public PixabayApiService $pixaBay;

  /**
   * Cache service.
   */
  protected CacheBackendInterface $cache;


  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\media_pixabay\Api\PixabayApiService $pixaBay
   *   The PixaBay API service.
   * @param \Drupal\Core\Config\ImmutableConfig $configFactory
   *   The config factory interface.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   Cache.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    PixabayApiService $pixaBay,
    ImmutableConfig $configFactory,
    CacheBackendInterface $cache,
  ) {
    $this->pixaBay = $pixaBay;
    $this->pixabayKeyConfig = $configFactory->get('media_pixabay_api');
    $this->cache = $cache;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('media_pixabay.api_service'),
      $container->get('config.factory')->get('media_pixabay.admin.config'),
      $container->get('cache.default'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function helpText() {
    return $this->t("Finds images based on a string field search.");
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, FormStateInterface $form, array $defaultValues = []) {
    $cardinality = $fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $defaultGeneration = $cardinality < 0 || $cardinality > 10 ? 10 : $cardinality;
    $textAmount = $cardinality == -1 ? 'unlimited' : $cardinality;

    $form['automator_image_pixabay_amount'] = [
      '#type' => 'number',
      '#title' => 'Generation Amount',
      '#description' => $this->t('Amount of images to fetch.', [
        '%amount' => $textAmount,
      ]),
      '#default_value' => $defaultValues['automator_image_pixabay_amount'] ?? $defaultGeneration,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $amount = $automatorConfig['image_pixabay_amount'] ?? 1;
    $values = [];
    if (!empty($entity->{$automatorConfig['base_field']}->value)) {
      foreach ($entity->get($automatorConfig['base_field']) as $wrapperEntity) {
        $searchTerm = $wrapperEntity->value;
        $cid = 'media_pixabay:' . $searchTerm;
        if ($cache = $this->cache->get($cid)) {
          $result = $cache->data;
        }
        else {
          $queryStr = UrlHelper::buildQuery([
            'key' => $this->pixabayKeyConfig,
            'q' => $searchTerm,
            'image_type' => 'image,illustration',
            'per_page' => 100,
          ]);
          $result = $this->pixaBay->apiCall($queryStr);
          // Cache like Pixabay module does.
          $this->cache->set($cid, $result, time() + 24 * 60 * 60, ['pixabay:' . $searchTerm]);
        }
        for ($i = 0; $i < $amount; $i++) {
          // At least Large Image should exists.
          if (!empty($result->hits[$i]->largeImageURL)) {
            $values[] = (array) $result->hits[$i];
          }
        }
      }
    }
    return $values;
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    if (is_array($value)) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $fileHelper = $this->getFileHelper();
    // Transform string to boolean.
    $fileEntities = [];

    // Successful counter, to only download as many as max.
    $successFul = 0;
    foreach ($values as $value) {
      // Get filename from previewURL.
      $fileName = basename($value['previewURL']) ?? 'created.jpg';
      // Everything validated, then we prepare the file path to save to.
      $filePath = $fileHelper->createFilePathFromFieldConfig($fileName, $fieldDefinition, $entity);
      // Get the right version.
      $binary = "";
      if (!empty($value['imageURL'])) {
        $binary = file_get_contents($value['imageURL']);
      }
      elseif (!empty($value['fullHDURL'])) {
        $binary = file_get_contents($value['fullHDURL']);
      }
      elseif (!empty($value['largeImageURL'])) {
        $binary = file_get_contents($value['largeImageURL']);
      }
      else {
        return;
      }
      // Create file entity from string.
      $image = $fileHelper->generateImageMetaDataFromBinary($binary, $filePath);
      // If we can save, we attach it.
      if ($image) {
        // Add to the entities list.
        $fileEntities[] = $image;

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality()) {
          break;
        }
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $fileEntities);
  }

}
