<?php

namespace Drupal\ai_automator_pixabay\Plugin\AiAutomatorType;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\ai_automators\PluginBaseClasses\ExternalBase;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;
use Drupal\media_pixabay\Api\PixabayApiService;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The PixaBay base class.
 */
class PixaBayBase extends ExternalBase implements AiAutomatorTypeInterface, ContainerFactoryPluginInterface {

  /**
   * The media Pixabay API key config.
   */
  protected string $pixabayKeyConfig;

  /**
   * If its media or not.
   *
   * @var bool
   */
  protected $isMedia = FALSE;

  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\media_pixabay\Api\PixabayApiService $pixaBay
   *   The PixaBay API service.
   * @param \Drupal\Core\Config\ImmutableConfig $configFactory
   *   The config factory interface.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   Cache.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    public PixabayApiService $pixaBay,
    protected ImmutableConfig $configFactory,
    protected CacheBackendInterface $cache,
  ) {
    $this->pixaBay = $pixaBay;
    $this->pixabayKeyConfig = $configFactory->get('media_pixabay_api');
    $this->cache = $cache;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('media_pixabay.api_service'),
      $container->get('config.factory')->get('media_pixabay.admin.config'),
      $container->get('cache.default'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function helpText() {
    return $this->t("Finds images based on a string field search.");
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, FormStateInterface $formState, array $defaultValues = []) {
    $cardinality = $fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $defaultGeneration = $cardinality < 0 || $cardinality > 10 ? 10 : $cardinality;
    $textAmount = $cardinality == -1 ? 'unlimited' : $cardinality;
    $fieldEntity = $entity;
    if ($this->isMedia) {
      $fieldEntity = $this->getGeneralHelper()->entityTypeManager()->getStorage('media')->create([
        'bundle' => 'image',
      ]);
    }
    if (!empty($defaultValues['automator_media_pixabay_type'])) {
      $fieldEntity = $this->getGeneralHelper()->entityTypeManager()->getStorage('media')->create([
        'bundle' => $defaultValues['automator_media_pixabay_type'],
      ]);
    }

    $form['automator_image_pixabay_amount'] = [
      '#type' => 'number',
      '#title' => 'Generation Amount',
      '#description' => $this->t('Amount of images to fetch.', [
        '%amount' => $textAmount,
      ]),
      '#default_value' => $defaultValues['automator_image_pixabay_amount'] ?? $defaultGeneration,
    ];

    $form['automator_image_pixabay_type_filter'] = [
      '#type' => 'checkboxes',
      '#title' => 'Image Type',
      '#description' => $this->t('Image Type to look for'),
      '#required' => TRUE,
      '#multiple' => TRUE,
      '#options' => [
        'image' => 'Image',
        'illustration' => 'Illustration',
      ],
      '#default_value' => $defaultValues['automator_image_pixabay_type_filter'] ?? [
        'image',
        'illustration',
      ],
    ];

    $form['meta_fields'] = [
      '#type' => 'details',
      '#title' => 'Meta Fields',
      '#open' => TRUE,
      '#description' => $this->t('Fields to fill out with the fetched data.'),
    ];

    $options = [
      '' => $this->t('-- Leave empty --'),
      'search_term' => $this->t('Search Term'),
      'type' => $this->t('Image Type (illustration, image)'),
      'user' => $this->t('Author of photo'),
      'tags' => $this->t('Tags of photo'),
      'page_url' => $this->t('Page URL'),
    ];
    $form['meta_fields']['automator_image_pixabay_alt_text'] = [
      '#type' => 'select',
      '#title' => 'Alt Text',
      '#description' => $this->t('Field to use for alt text field on the image'),
      '#options' => $options,
      '#default_value' => $defaultValues['automator_image_pixabay_alt_text'] ?? 'search_term',
    ];

    $form['meta_fields']['automator_image_pixabay_title'] = [
      '#type' => 'select',
      '#title' => 'Title',
      '#description' => $this->t('Field to use for title field on the image'),
      '#options' => $options,
      '#default_value' => $defaultValues['automator_image_pixabay_title'] ?? '',
    ];

    $link_options = $this->getGeneralHelper()->getFieldsOfType($fieldEntity, 'link');
    $form['meta_fields']['automator_image_pixabay_author_link'] = [
      '#type' => 'select',
      '#title' => 'Author Link',
      '#description' => $this->t('If you want to link to the author of the photo, connect it to a link field. This fills out the href and the username as title.'),
      '#empty_option' => $this->t('-- Leave empty --'),
      '#options' => $link_options,
      '#default_value' => $defaultValues['automator_image_pixabay_author_link'] ?? '',
    ];

    $form['meta_fields']['automator_image_pixabay_image_link'] = [
      '#type' => 'select',
      '#title' => 'Image Link',
      '#description' => $this->t('If you want to link to the image, connect it to a link field. This fills out the link image url without a title.'),
      '#empty_option' => $this->t('-- Leave empty --'),
      '#options' => $link_options,
      '#default_value' => $defaultValues['automator_image_pixabay_image_link'] ?? '',
    ];

    $list_options = $this->getGeneralHelper()->getFieldsOfType($fieldEntity, 'entity_reference', 'taxonomy_term');
    $form['meta_fields']['automator_image_pixabay_tags'] = [
      '#type' => 'select',
      '#title' => 'Tags',
      '#description' => $this->t('If you want to add tags to the image, connect it to a taxonomy field. Note that this will generate new taxonomy terms on the first connected vocabulary.'),
      '#empty_option' => $this->t('-- Leave empty --'),
      '#options' => $list_options,
      '#default_value' => $defaultValues['automator_image_pixabay_tags'] ?? '',
    ];

    $string_options = $this->getGeneralHelper()->getFieldsOfType($fieldEntity, 'string');
    $string_options = array_merge($string_options, $this->getGeneralHelper()->getFieldsOfType($fieldEntity, 'string_long'));
    $form['meta_fields']['automator_image_pixabay_author'] = [
      '#type' => 'select',
      '#title' => 'User',
      '#description' => $this->t('If you want to add the username of the author of the image, connect it to a string field.'),
      '#empty_option' => $this->t('-- Leave empty --'),
      '#options' => $string_options,
      '#default_value' => $defaultValues['automator_image_pixabay_author'] ?? '',
    ];

    $string_options = $this->getGeneralHelper()->getFieldsOfType($fieldEntity, 'string');
    $form['meta_fields']['automator_image_pixabay_page_id'] = [
      '#type' => 'select',
      '#title' => 'Page ID',
      '#description' => $this->t('If you want to add the page id of the image, connect it to a string field.'),
      '#empty_option' => $this->t('-- Leave empty --'),
      '#options' => $string_options,
      '#default_value' => $defaultValues['automator_image_pixabay_page_id'] ?? '',
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $amount = $automatorConfig['image_pixabay_amount'] ?? 1;
    $values = [];
    $t = 0;
    if (!empty($entity->{$automatorConfig['base_field']}->value)) {
      foreach ($entity->get($automatorConfig['base_field']) as $wrapperEntity) {
        $searchTerm = $wrapperEntity->value;
        $filters = !empty($automatorConfig['image_pixabay_type_filter']) ? implode(',', $automatorConfig['image_pixabay_type_filter']) : 'image,illustration';
        $cid = 'media_pixabay:' . $searchTerm . ':' . $filters;
        if ($cache = $this->cache->get($cid)) {
          $result = $cache->data;
        }
        else {
          $queryStr = UrlHelper::buildQuery([
            'key' => $this->pixabayKeyConfig,
            'q' => $searchTerm,
            'image_type' => $filters,
            'per_page' => 100,
          ]);
          $result = $this->pixaBay->apiCall($queryStr);
          // Cache like Pixabay module does.
          $this->cache->set($cid, $result, time() + 24 * 60 * 60, ['pixabay:' . $searchTerm]);
        }
        for ($i = 0; $i < $amount; $i++) {
          // At least Large Image should exists.
          if (!empty($result->hits[$i]->largeImageURL)) {
            $values[$t] = (array) $result->hits[$i];
            $values[$t]['name'] = $searchTerm;
            $t++;
          }
        }
      }
    }
    return $values;
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    if (is_array($value)) {
      return TRUE;
    }
    return FALSE;
  }

  /**
   * Helper function for alt text or title.
   *
   * @param string $field
   *   The field.
   * @param array $value
   *   The value back.
   *
   * @return string
   *   The value.
   */
  protected function getMetaFieldValue(string $field, array $value) {
    switch ($field) {
      case 'search_term':
        return $value['name'];

      case 'type':
        return $value['type'];

      case 'user':
        return $value['user'];

      case 'tags':
        return implode(', ', $value['tags']);

      case 'page_url':
        return $value['pageURL'];
    }
    return '';
  }

  /**
   * Helper function to fill in the extra data.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   * @param array $automatorConfig
   *   The automator config.
   * @param array $value
   *   The value.
   *
   * @return \Drupal\Core\Entity\ContentEntityInterface
   *   The entity.
   */
  public function fillExtraPixabayData(ContentEntityInterface $entity, array $automatorConfig, array $value): ContentEntityInterface {
    if ($automatorConfig['image_pixabay_author_link']) {
      $profileLink = 'https://pixabay.com/users/' . $value['user'] . '-' . $value['user_id'] . '/';
      $entity->get($automatorConfig['image_pixabay_author_link'])->uri = $profileLink;
      $entity->get($automatorConfig['image_pixabay_author_link'])->title = $value['user'];
    }
    if ($automatorConfig['image_pixabay_author']) {
      $entity->set($automatorConfig['image_pixabay_author'], $value['user']);
    }
    if ($automatorConfig['image_pixabay_image_link']) {
      $entity->get($automatorConfig['image_pixabay_image_link'])->uri = $value['largeImageURL'];
    }
    if ($automatorConfig['image_pixabay_tags']) {
      $tags = explode(',', $value['tags']);
      $tagEntities = [];
      $vocab = $this->getGeneralHelper()->getVocabulariesFromField($entity->getEntityTypeId(), $entity->bundle(), $automatorConfig['image_pixabay_tags']);
      if (!empty($vocab[0])) {
        foreach ($tags as $tag) {
          $tagEntities[] = $this->getGeneralHelper()->getOrGenerateTaxonomyTerm($vocab[0], $tag);
        }
        $entity->set($automatorConfig['image_pixabay_tags'], $tagEntities);
      }
    }
    if ($automatorConfig['image_pixabay_page_id']) {
      $entity->set($automatorConfig['image_pixabay_page_id'], $value['id']);
    }

    return $entity;
  }

}
