<?php

namespace Drupal\ai_automator_pixabay\Plugin\AiAutomatorType;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;

/**
 * The rules for an image field.
 */
#[AiAutomatorType(
  id: 'ai_automator_pixabay_image',
  label: new TranslatableMarkup('Pixabay: Image Fetcher'),
  field_rule: 'image',
  target: 'file',
)]
class PixaBayImage extends PixaBayBase implements AiAutomatorTypeInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Pixabay Image Fetcher';

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $fileHelper = $this->getFileHelper();
    // Transform string to boolean.
    $fileEntities = [];

    // Successful counter, to only download as many as max.
    $successFul = 0;
    foreach ($values as $value) {
      // Get filename from previewURL.
      $fileName = basename($value['previewURL']) ?? 'created.jpg';
      // Everything validated, then we prepare the file path to save to.
      $filePath = $fileHelper->createFilePathFromFieldConfig($fileName, $fieldDefinition, $entity);
      // Get the right version.
      $binary = "";
      if (!empty($value['imageURL'])) {
        $binary = file_get_contents($value['imageURL']);
      }
      elseif (!empty($value['fullHDURL'])) {
        $binary = file_get_contents($value['fullHDURL']);
      }
      elseif (!empty($value['largeImageURL'])) {
        $binary = file_get_contents($value['largeImageURL']);
      }
      else {
        return;
      }
      // Create file entity from string.
      $altText = $this->getMetaFieldValue($automatorConfig['image_pixabay_alt_text'], $value);
      $title = $this->getMetaFieldValue($automatorConfig['image_pixabay_title'], $value);
      $entity = $this->fillExtraPixabayData($entity, $automatorConfig, $value);
      $image = $fileHelper->generateImageMetaDataFromBinary($binary, $filePath, $altText, $title);
      // If we can save, we attach it.
      if ($image) {
        // Add to the entities list.
        $fileEntities[] = $image;

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality()) {
          break;
        }
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $fileEntities);
  }

}
