<?php

namespace Drupal\ai_automator_pixabay\Plugin\AiAutomatorType;

use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_automators\Attribute\AiAutomatorType;
use Drupal\ai_automators\PluginInterfaces\AiAutomatorTypeInterface;

/**
 * The rules for an media field.
 */
#[AiAutomatorType(
  id: 'ai_automator_pixabay_media',
  label: new TranslatableMarkup('Pixabay: Image Fetcher'),
  field_rule: 'entity_reference',
  target: 'media',
)]
class PixaBayMedia extends PixaBayBase implements AiAutomatorTypeInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'Pixabay: Image Fetcher';

  /**
   * If its media or not.
   *
   * @var bool
   */
  protected $isMedia = TRUE;

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, FormStateInterface $formState, $defaultValues = []) {
    $form = parent::extraFormFields($entity, $fieldDefinition, $formState, $defaultValues);

    $form['automator_image_pixabay_media_type'] = [
      '#type' => 'select',
      '#title' => 'Media Type',
      '#description' => $this->t('Media Type to set'),
      '#required' => TRUE,
      '#options' => $this->getFileHelper()->getMediaBundles(),
      '#default_value' => $defaultValues['automator_image_pixabay_media_type'] ?? 'image',
      '#weight' => 0,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition, array $automatorConfig) {
    $fileHelper = $this->getFileHelper();
    $mediaType = $automatorConfig['image_pixabay_media_type'];

    // Transform string to boolean.
    $mediaEntitities = [];

    // Successful counter, to only download as many as max.
    $successFul = 0;
    foreach ($values as $value) {
      // Get filename from previewURL.
      $fileName = basename($value['previewURL']) ?? 'created.jpg';
      // Get the right version.
      $binary = "";
      if (!empty($value['imageURL'])) {
        $binary = file_get_contents($value['imageURL']);
      }
      elseif (!empty($value['fullHDURL'])) {
        $binary = file_get_contents($value['fullHDURL']);
      }
      elseif (!empty($value['largeImageURL'])) {
        $binary = file_get_contents($value['largeImageURL']);
      }
      else {
        return;
      }
      // Create file entity from string.
      $altText = $this->getMetaFieldValue($automatorConfig['image_pixabay_alt_text'], $value);
      $title = $this->getMetaFieldValue($automatorConfig['image_pixabay_title'], $value);

      // Create file entity from string.
      $media = $fileHelper->generateMediaImageFromFile($fileName, $binary, $mediaType, $value['name'], [
        'alt' => $altText,
        'title' => $title,
      ]);
      // Fill extra data and save.
      $media = $this->fillExtraPixabayData($media, $automatorConfig, $value);
      $media->save();
      // If we can save, we attach it.
      if ($media) {
        $mediaEntitities[] = $media->id();

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality()) {
          break;
        }
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $mediaEntitities);
  }

}
