<?php

namespace Drupal\ai_ckeditor_wcag\Plugin\AICKEditor;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\AiVdbProviderPluginManager;
use Drupal\ai_ckeditor\AiCKEditorPluginBase;
use Drupal\ai_ckeditor\Attribute\AiCKEditor;
use Drupal\ai_ckeditor\Command\AiRequestCommand;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin to check selected text against WCAG compatibility.
 */
#[AiCKEditor(
  id: 'ai_ckeditor_wcag',
  label: new TranslatableMarkup('WCAG'),
  description: new TranslatableMarkup('Check CWAG compatibility of selected text.'),
)]
final class WCAG extends AiCKEditorPluginBase {

  /**
   * The AI VDB Provider.
   *
   * @var \Drupal\ai\AiVdbProviderPluginManager
   */
  protected AiVdbProviderPluginManager $vdbProviderManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->vdbProviderManager = $container->get('ai.vdb_provider');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'provider' => NULL,
      'wcag_version' => NULL,
      'wcag_level' => NULL,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state) {
    // If a subform is received, we want the full form state.
    if ($form_state instanceof SubformStateInterface) {
      $form_state = $form_state->getCompleteFormState();
    }

    $options = $this->aiProviderManager->getSimpleProviderModelOptions('chat');
    array_shift($options);
    array_splice($options, 0, 1);

    $form['provider'] = [
      '#type' => 'select',
      "#empty_option" => $this->t('-- Default from AI module (chat) --'),
      '#title' => $this->t('AI provider'),
      '#options' => $options,
      '#default_value' => $this->configuration['provider'] ?? $this->aiProviderManager->getSimpleDefaultProviderOptions('chat'),
      '#description' => $this->t('Select which provider to use for this plugin. See the <a href=":link">Provider overview</a> for details about each provider.', [':link' => '/admin/config/ai/providers']),
    ];

    $form['wcag_version'] = [
      '#type' => 'select',
      '#title' => $this->t('WCAG version'),
      '#options' => [
        '1.0' => $this->t('1.0'),
        '2.0' => $this->t('2.0'),
        '2.1' => $this->t('2.1'),
        '2.2' => $this->t('2.2'),
      ],
      '#default_value' => $this->configuration['wcag_version'] ?? '2.2',
      '#description' => $this->t('Select the WCAG version to check against.'),
    ];

    $form['wcag_level'] = [
      '#type' => 'select',
      '#title' => $this->t('WCAG level'),
      '#options' => [
        'A' => $this->t('A'),
        'AA' => $this->t('AA'),
        'AAA' => $this->t('AAA'),
      ],
      '#default_value' => $this->configuration['wcag_level'] ?? 'AAA',
      '#description' => $this->t('Select the WCAG level to check against.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {}

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $this->configuration['provider'] = $form_state->getValue('provider');
    $this->configuration['wcag_version'] = $form_state->getValue('wcag_version');
    $this->configuration['wcag_level'] = $form_state->getValue('wcag_level');
  }

  /**
   * {@inheritdoc}
   */
  public function buildCkEditorModalForm(array $form, FormStateInterface $form_state, array $settings = []) {
    $storage = $form_state->getStorage();
    $editor_id = $this->requestStack->getParentRequest()->get('editor_id');

    if (empty($storage['selected_text'])) {
      return [
        '#markup' => '<p>' . $this->t('You must select some text before you can check the WCAG compatibility.') . '</p>',
      ];
    }

    $form = parent::buildCkEditorModalForm($form, $form_state);

    $form['selected_text'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Selected text to check'),
      '#disabled' => TRUE,
      '#default_value' => $storage['selected_text'],
    ];

    $form['response_text'] = [
      '#type' => 'text_format',
      '#title' => $this->t('AI response'),
      '#description' => $this->t('The response from AI will appear in the box above.'),
      '#prefix' => '<div id="ai-ckeditor-response">',
      '#suffix' => '</div>',
      '#default_value' => '',
      '#allowed_formats' => [$editor_id],
      '#format' => $editor_id,
    ];

    $form['actions']['generate']['#value'] = $this->t('Check the selected text');

    // Remove the submit button.
    $form['actions']['submit'] = [];

    return $form;
  }

  /**
   * Generate text callback.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return mixed
   *   The result of the AJAX operation.
   */
  public function ajaxGenerate(array &$form, FormStateInterface $form_state) {
    $values = $form_state->getValues();

    try {
      $prompt = "Check if the content of the following text is in line with WCAG version " . $this->configuration['wcag_version'] . " level " . $this->configuration['wcag_level'] . " rules: " . $values["plugin_config"]["selected_text"];
      $prompt .= "\n\nRespond in the same language as the text.";

      $response = new AjaxResponse();
      $values = $form_state->getValues();
      $response->addCommand(new AiRequestCommand($prompt, $values["editor_id"], $this->pluginDefinition['id'], 'ai-ckeditor-response'));
      return $response;
    }
    catch (\Exception $e) {
      $this->logger->error("There was an error in the AI CKEditor WCAG plugin.");
      return $form['plugin_config']['response_text']['#value'] = "There was an error in the AI CKEditor WCAG plugin.";
    }
  }

}
