<?php

namespace Drupal\ai_content_translation\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\ai_content_translation\Service\OpenAITranslationService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Url;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Link;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Controller for AI translation functionality.
 */
class AITranslationController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The OpenAI translation service.
   *
   * @var \Drupal\ai_content_translation\Service\OpenAITranslationService
   */
  protected $translationService;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AITranslationController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\ai_content_translation\Service\OpenAITranslationService $translation_service
   *   The OpenAI translation service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    OpenAITranslationService $translation_service,
    LanguageManagerInterface $language_manager,
    MessengerInterface $messenger,
    ConfigFactoryInterface $config_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->translationService = $translation_service;
    $this->languageManager = $language_manager;
    $this->messenger = $messenger;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('ai_content_translation.openai_translation'),
      $container->get('language_manager'),
      $container->get('messenger'),
      $container->get('config.factory')
    );
  }
  
  /**
   * Logs a message respecting the configured log settings.
   *
   * @param string $level
   *   The log level ('error', 'notice', 'info')
   * @param string $message
   *   The message to log.
   * @param array $context
   *   The context parameters.
   */
  protected function log($level, $message, array $context = []) {
    $config = $this->configFactory->get('ai_content_translation.settings');
    $enable_logging = $config->get('enable_logging') ?? TRUE;
    $configured_level = $config->get('log_level') ?: 'notice';
    
    // Always log errors regardless of settings
    if ($level === 'error') {
      \Drupal::logger('ai_content_translation')->error($message, $context);
      return;
    }
    
    // Skip if logging is disabled
    if (!$enable_logging) {
      return;
    }
    
    // Check if the current log level should be logged based on config
    $should_log = FALSE;
    switch ($configured_level) {
      case 'error':
        // Only errors, which we already handled above
        $should_log = FALSE;
        break;
      
      case 'notice':
        // Errors and notices
        $should_log = ($level === 'notice');
        break;
        
      case 'info':
        // All levels
        $should_log = TRUE;
        break;
    }
    
    if ($should_log) {
      \Drupal::logger('ai_content_translation')->$level($message, $context);
    }
  }
  
  /**
   * Checks if a value should be translated.
   *
   * @param mixed $value
   *   The value to check.
   *
   * @return bool
   *   TRUE if the value should be translated, FALSE otherwise.
   */
  protected function shouldTranslate($value) {
    // Skip NULL values
    if ($value === NULL) {
      return FALSE;
    }
    
    // Skip empty strings or strings that only contain whitespace
    if (is_string($value) && trim($value) === '') {
      return FALSE;
    }
    
    return TRUE;
  }

  /**
   * Translates image field alt and title attributes.
   *
   * @param object $field
   *   The image field to process.
   * @param \Drupal\Core\Entity\ContentEntityInterface $translation
   *   The entity translation.
   * @param string $field_name
   *   The name of the field.
   * @param string $language_name
   *   The human-readable target language name.
   */
  protected function translateImageAttributes($field, ContentEntityInterface $translation, $field_name, $language_name) {
    $this->log('notice', 'Translating image attributes for field: @field_name', [
      '@field_name' => $field_name,
    ]);
    
    foreach ($field as $delta => $item) {
      $updated_item = [
        'target_id' => $item->target_id,
        'width' => $item->width ?? NULL,
        'height' => $item->height ?? NULL,
      ];
      
      // Translate alt text if it exists
      if (isset($item->alt) && $this->shouldTranslate($item->alt)) {
        $this->log('info', 'Translating ALT text for image in field: @field_name', [
          '@field_name' => $field_name,
        ]);
        $translated_alt = $this->translationService->translateText($item->alt, $language_name);
        $updated_item['alt'] = $translated_alt;
      } else {
        $updated_item['alt'] = $item->alt ?? '';
      }
      
      // Translate title text if it exists
      if (isset($item->title) && $this->shouldTranslate($item->title)) {
        $this->log('info', 'Translating title text for image in field: @field_name', [
          '@field_name' => $field_name,
        ]);
        $translated_title = $this->translationService->translateText($item->title, $language_name);
        $updated_item['title'] = $translated_title;
      } else {
        $updated_item['title'] = $item->title ?? '';
      }
      
      // Keep other attributes if they exist
      if (isset($item->display)) {
        $updated_item['display'] = $item->display;
      }
      
      // Set the updated values back to the translation
      $translation->get($field_name)->set($delta, $updated_item);
    }
  }

  /**
   * Recursively translates all content entities including paragraphs.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to translate.
   * @param string $target_lang
   *   The target language code.
   * @param string $language_name
   *   The human-readable language name.
   * @param array $processed_entities
   *   Array of already processed entity IDs to prevent infinite recursion.
   * @param \Drupal\Core\Entity\ContentEntityInterface $root_entity
   *   The root entity (node) being translated, used to verify ownership.
   */
  protected function translateContentEntities(ContentEntityInterface $entity, $target_lang, $language_name, array &$processed_entities = [], ContentEntityInterface $root_entity = NULL) {
    // Set root entity to the current entity if not provided (first call)
    if ($root_entity === NULL) {
      $root_entity = $entity;
    }
    
    // Generate a unique ID for this entity to prevent recursion
    $entity_id = $entity->getEntityTypeId() . ':' . $entity->id();
    
    // Log the entity being processed
    $this->log('notice', 'Processing entity: @type #@id', [
      '@type' => $entity->getEntityTypeId(),
      '@id' => $entity->id(),
    ]);
    
    // Skip if already processed
    if (in_array($entity_id, $processed_entities)) {
      $this->log('info', 'Skipping already processed entity: @entity_id', [
        '@entity_id' => $entity_id,
      ]);
      return;
    }
    
    // Skip file, media, and taxonomy term entities completely
    if (in_array($entity->getEntityTypeId(), ['file', 'media', 'taxonomy_term'])) {
      $this->log('info', 'Skipping file/media/taxonomy entity: @type #@id', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
      ]);
      return;
    }
    
    // Mark this entity as processed
    $processed_entities[] = $entity_id;
    
    // Get all fields to find entity references
    foreach ($entity->getFields() as $field_name => $field) {
      // Skip internal fields and non-entity reference fields
      if (in_array($field_name, ['langcode', 'default_langcode', 'content_translation_source', 
          'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
          'content_translation_changed', 'revision_uid', 'revision_timestamp', 'revision_log'])) {
        continue;
      }
      
      $field_definition = $field->getFieldDefinition();
      $field_type = $field_definition->getType();
      
      // Check for image fields and translate alt/title attributes if present
      if ($field_type === 'image') {
        if ($entity->hasTranslation($target_lang)) {
          $translation = $entity->getTranslation($target_lang);
          $this->translateImageAttributes($field, $translation, $field_name, $language_name);
        }
        continue;
      }
      
      // Handle entity reference fields (including paragraphs and other content entities)
      if ($field_type == 'entity_reference' || $field_type == 'entity_reference_revisions') {
        $target_type = $field_definition->getSetting('target_type');
        
        $this->log('info', 'Found reference field: @field_name, target type: @target_type', [
          '@field_name' => $field_name,
          '@target_type' => $target_type,
        ]);
        
        // Skip file, media, and taxonomy term field references
        if (in_array($target_type, ['file', 'media', 'taxonomy_term'])) {
          $this->log('info', 'Skipping file/media/taxonomy reference field: @field_name', [
            '@field_name' => $field_name,
          ]);
          continue;
        }
        
        // Process all referenced entities
        foreach ($field as $delta => $item) {
          if ($item->entity && $item->entity instanceof ContentEntityInterface) {
            $referenced_entity = $item->entity;
            
            $this->log('info', 'Found referenced entity: @type #@id in field @field_name', [
              '@type' => $referenced_entity->getEntityTypeId(),
              '@id' => $referenced_entity->id(),
              '@field_name' => $field_name,
            ]);
            
            // Skip file, media, and taxonomy term entities
            if (in_array($referenced_entity->getEntityTypeId(), ['file', 'media', 'taxonomy_term'])) {
              $this->log('info', 'Skipping file/media/taxonomy referenced entity: @type #@id', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
              ]);
              continue;
            }
            
            // Verify this entity belongs to our node by checking the parent entity
            $belongs_to_current_node = FALSE;
            
            // For paragraphs, check if parent entity matches our root node
            if ($referenced_entity->getEntityTypeId() == 'paragraph') {
              // If paragraph has a parent_field_name and parent_type property, check them
              if (method_exists($referenced_entity, 'getParentEntity')) {
                $parent = $referenced_entity->getParentEntity();
                if ($parent) {
                  // Either the parent is our root entity or a descendant
                  if ($parent->id() == $root_entity->id() && $parent->getEntityTypeId() == $root_entity->getEntityTypeId()) {
                    $belongs_to_current_node = TRUE;
                    $this->log('info', 'Paragraph @id belongs to root entity directly', [
                      '@id' => $referenced_entity->id(),
                    ]);
                  }
                  // Or the parent could be another paragraph that belongs to our root entity
                  else if ($parent->getEntityTypeId() == 'paragraph') {
                    // We're already processing this, so it must belong to our structure
                    $belongs_to_current_node = TRUE;
                    $this->log('info', 'Paragraph @id belongs to root entity via another paragraph', [
                      '@id' => $referenced_entity->id(),
                    ]);
                  }
                }
              } else {
                // Fallback for older Paragraph module versions
                // Since we're recursively following references from the root node,
                // any paragraph we find must be part of the current node's structure
                $belongs_to_current_node = TRUE;
                $this->log('info', 'Paragraph @id belongs to structure (fallback method)', [
                  '@id' => $referenced_entity->id(),
                ]);
              }
            } else {
              // For non-paragraph entities, we trust the reference structure
              // If we found it by traversing from the root node, it belongs to the node
              $belongs_to_current_node = TRUE;
              $this->log('info', 'Non-paragraph entity @type #@id belongs to structure', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
              ]);
            }
            
            // Only process if this entity belongs to our node
            if ($belongs_to_current_node && $referenced_entity->isTranslatable()) {
              $this->log('notice', 'Entity @type #@id will be translated to @lang', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
                '@lang' => $target_lang,
              ]);
              
              // Create a translation if it doesn't exist
              if (!$referenced_entity->hasTranslation($target_lang)) {
                $referenced_translation = $referenced_entity->addTranslation($target_lang, $referenced_entity->toArray());
                
                // Process all fields in the referenced entity
                foreach ($referenced_entity->getFields() as $ref_field_name => $ref_field) {
                  if (in_array($ref_field_name, ['langcode', 'default_langcode', 'content_translation_source', 
                      'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
                      'content_translation_changed', 'revision_uid', 'revision_timestamp', 'revision_log'])) {
                    continue;
                  }
                  
                  $ref_field_type = $ref_field->getFieldDefinition()->getType();
                  
                  // Skip taxonomy field references in referenced entity
                  $ref_target_type = NULL;
                  if ($ref_field_type === 'entity_reference' || $ref_field_type === 'entity_reference_revisions') {
                    $ref_target_type = $ref_field->getFieldDefinition()->getSetting('target_type');
                    if ($ref_target_type === 'taxonomy_term') {
                      $this->log('info', 'Skipping taxonomy reference field in referenced entity: @field_name', [
                        '@field_name' => $ref_field_name,
                      ]);
                      continue;
                    }
                  }
                  
                  // Handle image fields for alt and title translation
                  if ($ref_field_type === 'image') {
                    $this->log('info', 'Translating image field: @field_name in referenced entity', [
                      '@field_name' => $ref_field_name,
                    ]);
                    $this->translateImageAttributes($ref_field, $referenced_translation, $ref_field_name, $language_name);
                    continue;
                  }
                  
                  // Skip all non-text fields (only translate text fields)
                  if (!in_array($ref_field_type, ['text', 'text_long', 'text_with_summary', 'string', 'string_long'])) {
                    $this->log('info', 'Skipping non-text field: @field_name (type: @type)', [
                      '@field_name' => $ref_field_name,
                      '@type' => $ref_field_type,
                    ]);
                    continue;
                  }
                  
                  // Skip datetime, time, numeric, and currency fields in referenced entities
                  if (in_array($ref_field_type, ['datetime', 'daterange', 'timestamp', 'time', 'integer', 'decimal', 'float', 'numeric', 'number', 'list_integer', 'list_float', 'list_decimal'])) {
                    $this->log('info', 'Skipping datetime/numeric field in referenced entity: @field_name (type: @type)', [
                      '@field_name' => $ref_field_name,
                      '@type' => $ref_field_type,
                    ]);
                    continue;
                  }
                  
                  // Translate text fields only
                  $this->log('info', 'Translating text field: @field_name', [
                    '@field_name' => $ref_field_name,
                  ]);
                  
                  foreach ($ref_field as $ref_delta => $ref_item) {
                    if ($this->shouldTranslate($ref_item->value)) {
                      $translated_value = $this->translationService->translateText($ref_item->value, $language_name);
                      $referenced_translation->get($ref_field_name)->set($ref_delta, [
                        'value' => $translated_value, 
                        'format' => $ref_item->format ?? NULL
                      ]);
                    } else {
                      $this->log('info', 'Skipping empty value in field: @field_name', [
                        '@field_name' => $ref_field_name,
                      ]);
                    }
                  }
                }
                
                // Save the referenced entity translation
                $referenced_translation->save();
                
                $this->log('notice', 'Saved translation for entity @type #@id to @lang', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                  '@lang' => $target_lang,
                ]);
                
                // Recursively process nested entities - pass the root entity to maintain context
                $this->translateContentEntities($referenced_entity, $target_lang, $language_name, $processed_entities, $root_entity);
              } else {
                $this->log('info', 'Entity @type #@id already has a @lang translation', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                  '@lang' => $target_lang,
                ]);
              }
            } else {
              if (!$belongs_to_current_node) {
                $this->log('info', 'Entity @type #@id does not belong to the current node structure', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                ]);
              }
              if (!$referenced_entity->isTranslatable()) {
                $this->log('info', 'Entity @type #@id is not translatable', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                ]);
              }
            }
          }
        }
      }
    }
    
    $this->log('notice', 'Completed processing entity: @type #@id', [
      '@type' => $entity->getEntityTypeId(),
      '@id' => $entity->id(),
    ]);
  }

  /**
   * Translates an entity using AI.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param mixed $entity_id
   *   The entity.
   * @param string $target_lang
   *   The target language code.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   */
  public function translate($entity_type, $entity_id, $target_lang) {
    // Get the entity.
    if (is_object($entity_id)) {
      $entity = $entity_id;
    }
    else {
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    }

    if (!$entity) {
      $this->messenger->addError($this->t('Entity not found.'));
      return $this->redirect('<front>');
    }

    // Skip translation for taxonomy terms
    if ($entity_type === 'taxonomy_term') {
      $this->messenger->addError($this->t('Taxonomy terms are not supported for AI translation.'));
      return $this->redirect('<front>');
    }

    if (!$entity instanceof ContentEntityInterface || !$entity->isTranslatable()) {
      $this->messenger->addError($this->t('Entity is not translatable.'));
      return $this->redirect('<front>');
    }

    // Get the language object.
    $language = $this->languageManager->getLanguage($target_lang);
    if (!$language) {
      $this->messenger->addError($this->t('Invalid target language.'));
      return $this->redirect('<front>');
    }

    // Check if translation already exists.
    if ($entity->hasTranslation($target_lang)) {
      $this->messenger->addWarning($this->t('Translation already exists for @language.', [
        '@language' => $language->getName(),
      ]));
      return $this->redirect('entity.' . $entity_type . '.content_translation_overview', [
        $entity_type => $entity->id(),
      ]);
    }

    try {
      $this->log('notice', 'Starting AI translation of @type #@id to @language', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@language' => $language->getName(),
      ]);
      
      // Create a new translation.
      $translation = $entity->addTranslation($target_lang, $entity->toArray());
      
      // Translate title if it exists (title is a text field).
      if ($entity->hasField('title')) {
        $title_value = $entity->get('title')->value;
        if ($this->shouldTranslate($title_value)) {
          $translated_title = $this->translationService->translateText($title_value, $language->getName());
          $translation->set('title', $translated_title);
          $this->log('info', 'Translated title field');
        } else {
          $this->log('info', 'Skipping empty title field');
        }
      }

      // Translate body if it exists (body is a text field).
      if ($entity->hasField('body')) {
        $body_value = $entity->get('body')->value;
        $body_format = $entity->get('body')->format;
        
        if ($this->shouldTranslate($body_value)) {
          $translated_body = $this->translationService->translateText($body_value, $language->getName());
          $translation->set('body', [
            'value' => $translated_body,
            'format' => $body_format,
          ]);
          $this->log('info', 'Translated body field');
        } else {
          $this->log('info', 'Skipping empty body field');
        }
      }

      // Translate all text fields that might contain content.
      foreach ($entity->getFields() as $field_name => $field) {
        // Skip already translated fields
        if (in_array($field_name, ['title', 'body', 'langcode', 'default_langcode', 'content_translation_source', 
            'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
            'content_translation_changed'])) {
          continue;
        }

        $field_type = $field->getFieldDefinition()->getType();
        
        // Skip taxonomy fields
        if ($field_type === 'entity_reference' || $field_type === 'entity_reference_revisions') {
          $target_type = $field->getFieldDefinition()->getSetting('target_type');
          if ($target_type === 'taxonomy_term') {
            $this->log('info', 'Skipping taxonomy term reference field on main entity: @field_name', [
              '@field_name' => $field_name,
            ]);
            continue;
          }
        }
        
        // Handle image fields to translate alt and title text
        if ($field_type === 'image') {
          $this->log('info', 'Processing image field for alt/title translation: @field_name', [
            '@field_name' => $field_name,
          ]);
          $this->translateImageAttributes($field, $translation, $field_name, $language->getName());
          continue;
        }
        
        // Skip non-text fields
        if (!in_array($field_type, ['text', 'text_long', 'text_with_summary', 'string', 'string_long'])) {
          $this->log('info', 'Skipping non-text field on main entity: @field_name (type: @type)', [
            '@field_name' => $field_name,
            '@type' => $field_type,
          ]);
          continue;
        }
        
        // Skip datetime, time, numeric, and currency fields
        if (in_array($field_type, ['datetime', 'daterange', 'timestamp', 'time', 'integer', 'decimal', 'float', 'numeric', 'number', 'list_integer', 'list_float', 'list_decimal'])) {
          $this->log('info', 'Skipping datetime/numeric field on main entity: @field_name (type: @type)', [
            '@field_name' => $field_name,
            '@type' => $field_type,
          ]);
          continue;
        }

        // Handle text fields only
        $this->log('info', 'Translating text field on main entity: @field_name', [
          '@field_name' => $field_name,
        ]);
        
        foreach ($field as $delta => $item) {
          if ($this->shouldTranslate($item->value)) {
            $translated_value = $this->translationService->translateText($item->value, $language->getName());
            $translation->get($field_name)->set($delta, ['value' => $translated_value, 'format' => $item->format ?? NULL]);
          } else {
            $this->log('info', 'Skipping empty value in field: @field_name', [
              '@field_name' => $field_name,
            ]);
          }
        }
      }

      // Initialize the array for tracking processed entities
      $processed_entities = [];
      
      // Recursively process and translate all paragraph and content entities
      // Pass the entity as the root entity to ensure we only translate entities belonging to this node
      $this->translateContentEntities($entity, $target_lang, $language->getName(), $processed_entities, $entity);

      // Save the entity with the new translation.
      $translation->save();
      
      $this->log('notice', 'Completed AI translation of @type #@id to @language', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@language' => $language->getName(),
      ]);


      // Get entity title for the success message
      $entity_title = '';
      if ($entity->hasField('title') && !empty($entity->get('title')->value)) {
        $entity_title = $entity->get('title')->value;
      }
      elseif ($entity->hasField('name') && !empty($entity->get('name')->value)) {
        $entity_title = $entity->get('name')->value;
      }
      elseif ($entity->hasField('label') && !empty($entity->get('label')->value)) {
        $entity_title = $entity->get('label')->value;
      }

      // Create links for the success message
      $view_url = $entity->toUrl('canonical', ['language' => $language]);
      $view_link = Link::fromTextAndUrl($this->t('View translation'), $view_url)->toString();

      $edit_url = $entity->toUrl('edit-form', ['language' => $language]);
      $edit_link = Link::fromTextAndUrl($this->t('Edit translation'), $edit_url)->toString();

      // Add success message with title and links
      if (!empty($entity_title)) {
        $this->messenger->addStatus($this->t('AI translation of "@title" to @language has been generated. @view_link | @edit_link', [
          '@title' => $entity_title,
          '@language' => $language->getName(),
          '@view_link' => $view_link,
          '@edit_link' => $edit_link,
        ]));
      } else {
        $this->messenger->addStatus($this->t('AI translation to @language has been generated. !view_link | !edit_link', [
          '@language' => $language->getName(),
          '@view_link' => $view_link,
          '@edit_link' => $edit_link,
        ]));
      }
      // Redirect back to the referring page (preserving query parameters/filters)
      $request = \Drupal::request();
      $referer = $request->headers->get('referer');
      
      if ($referer) {
        // User is coming from a specific page, redirect back there
        return new RedirectResponse($referer);
      } 
      else {
        // Fallback to the content overview page if referer is not available
        return $this->redirect('system.admin_content');
      }
    }
    catch (\Exception $e) {
      $this->log('error', 'Error generating AI translation for @type #@id: @error', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@error' => $e->getMessage(),
      ]);
      
      $this->messenger->addError($this->t('Error generating AI translation: @error', [
        '@error' => $e->getMessage(),
      ]));
      
      return $this->redirect('entity.' . $entity_type . '.content_translation_overview', [
        $entity_type => $entity->id(),
      ]);
    }
  }
}