<?php

namespace Drupal\ai_content_translation\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for AI Content Translation settings.
 */
class AIContentTranslationSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_content_translation_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_content_translation.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_content_translation.settings');

    $form['api_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('OpenAI API Settings'),
      '#open' => TRUE,
    ];

    $form['api_settings']['api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('OpenAI API Key'),
      '#default_value' => $config->get('api_key'),
      '#required' => TRUE,
      '#size' => 100,
      '#maxlength' => 200,
      '#description' => $this->t('Enter your OpenAI API key. You can get one from <a href="https://platform.openai.com/api-keys" target="_blank">OpenAI API Keys</a>.'),
    ];

    $form['api_settings']['model'] = [
      '#type' => 'select',
      '#title' => $this->t('OpenAI Model'),
      '#options' => [
        'gpt-4' => 'GPT-4',
        'gpt-4-turbo' => 'GPT-4 Turbo',
        'gpt-3.5-turbo' => 'GPT-3.5 Turbo',
      ],
      '#default_value' => $config->get('model') ?: 'gpt-4',
      '#description' => $this->t('Select the OpenAI model to use for translation.'),
    ];

    $form['prompt_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Translation Prompt Settings'),
      '#open' => TRUE,
    ];
    
    $form['prompt_settings']['system_prompt'] = [
      '#type' => 'textarea',
      '#title' => $this->t('System Prompt'),
      '#default_value' => $config->get('system_prompt') ?: $this->t('You are a professional translator. Your task is to translate the content provided while preserving the original meaning, tone, and formatting including all the HTML tags. Provide only the translated text without explanations.'),
      '#description' => $this->t('Enter the system prompt to be used for translation. This prompt will guide the AI on how to translate your content.'),
      '#rows' => 5,
    ];
    
    $form['prompt_settings']['temperature'] = [
      '#type' => 'number',
      '#title' => $this->t('Temperature'),
      '#default_value' => $config->get('temperature') ?: 0.3,
      '#min' => 0,
      '#max' => 2,
      '#step' => 0.1,
      '#description' => $this->t('Controls randomness: lower values (e.g. 0.2) make translations more deterministic, higher values (e.g. 0.8) make translations more creative. Recommended: 0.3 for translations.'),
    ];

    $form['advanced_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Advanced Settings'),
      '#open' => FALSE,
    ];

    $form['advanced_settings']['timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Request Timeout (seconds)'),
      '#default_value' => $config->get('timeout') ?: 180,
      '#min' => 30,
      '#max' => 600,
      '#description' => $this->t('Maximum time in seconds to wait for a translation response. For large content, use a higher value (180-300 seconds recommended).'),
    ];

    $form['advanced_settings']['connect_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('Connection Timeout (seconds)'),
      '#default_value' => $config->get('connect_timeout') ?: 60,
      '#min' => 5,
      '#max' => 600,
      '#description' => $this->t('Maximum time in seconds to wait for establishing connection to the API. Default: 60 seconds.'),
    ];
    
    $form['debug_settings'] = [
      '#type' => 'details',
      '#title' => $this->t('Debugging & Logging'),
      '#open' => FALSE,
    ];
    
    $form['debug_settings']['enable_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable detailed logging'),
      '#default_value' => $config->get('enable_logging') ?? TRUE,
      '#description' => $this->t('When enabled, the module will log detailed information about translations, including text samples. Disable in production for better performance and privacy.'),
    ];
    
    $form['debug_settings']['log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Log Detail Level'),
      '#options' => [
        'error' => $this->t('Errors Only'),
        'notice' => $this->t('Normal (Errors + Completion notices)'),
        'info' => $this->t('Verbose (All details)'),
      ],
      '#default_value' => $config->get('log_level') ?: 'notice',
      '#description' => $this->t('Controls how much detail is logged when logging is enabled.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_logging"]' => ['checked' => TRUE],
        ],
      ],
    ];
    
    $form['debug_settings']['show_sample_text'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Log sample text in logs'),
      '#default_value' => $config->get('show_sample_text') ?? TRUE,
      '#description' => $this->t('Include snippets of source and translated text in logs. Disable if your content contains sensitive information.'),
      '#states' => [
        'visible' => [
          ':input[name="enable_logging"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('ai_content_translation.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('system_prompt', $form_state->getValue('system_prompt'))
      ->set('model', $form_state->getValue('model'))
      ->set('temperature', $form_state->getValue('temperature'))
      ->set('timeout', $form_state->getValue('timeout'))
      ->set('connect_timeout', $form_state->getValue('connect_timeout'))
      ->set('enable_logging', $form_state->getValue('enable_logging'))
      ->set('log_level', $form_state->getValue('log_level'))
      ->set('show_sample_text', $form_state->getValue('show_sample_text'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}