<?php
namespace Drupal\ai_content_translation\Service;
use Drupal\Core\Config\ConfigFactoryInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\StringTranslation\StringTranslationTrait;
/**
 * Service for translating content using OpenAI.
 */
class OpenAITranslationService {
  use StringTranslationTrait;
  /**
   * Config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;
  /**
   * HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;
  /**
   * Constructs a new OpenAITranslationService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(ConfigFactoryInterface $config_factory, ClientInterface $http_client) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
  }
  
  /**
   * Logs a message respecting the configured log settings.
   *
   * @param string $level
   *   The log level ('error', 'notice', 'info')
   * @param string $message
   *   The message to log.
   * @param array $context
   *   The context parameters.
   */
  protected function log($level, $message, array $context = []) {
    $config = $this->configFactory->get('ai_content_translation.settings');
    $enable_logging = $config->get('enable_logging') ?? TRUE;
    $configured_level = $config->get('log_level') ?: 'notice';
    
    // Always log errors regardless of settings
    if ($level === 'error') {
      \Drupal::logger('ai_content_translation')->error($message, $context);
      return;
    }
    
    // Skip if logging is disabled
    if (!$enable_logging) {
      return;
    }
    
    // Check if the current log level should be logged based on config
    $should_log = FALSE;
    switch ($configured_level) {
      case 'error':
        // Only errors, which we already handled above
        $should_log = FALSE;
        break;
      
      case 'notice':
        // Errors and notices
        $should_log = ($level === 'notice');
        break;
        
      case 'info':
        // All levels
        $should_log = TRUE;
        break;
    }
    
    if ($should_log) {
      \Drupal::logger('ai_content_translation')->$level($message, $context);
    }
  }
  
  /**
   * Translates text using OpenAI.
   *
   * @param string $text
   *   The text to translate.
   * @param string $target_language
   *   The target language name.
   *
   * @return string
   *   The translated text.
   */
  public function translateText($text, $target_language) {
    $config = $this->configFactory->get('ai_content_translation.settings');
    $api_key = $config->get('api_key');
    $system_prompt = $config->get('system_prompt');
    $model = $config->get('model') ?: 'gpt-4';
    
    // Get configurable parameters with defaults
    $temperature = $config->get('temperature') ?: 0.3;
    $timeout = $config->get('timeout') ?: 180;
    $connect_timeout = $config->get('connect_timeout') ?: 30;
    $show_sample_text = $config->get('show_sample_text') ?? TRUE;
    
    if (empty($api_key)) {
      throw new \Exception($this->t('OpenAI API key is not configured.'));
    }
    
    // Log the first 100 characters of the source text to be translated
    if ($show_sample_text) {
      $truncated_source = mb_substr($text, 0, 100);
      if (mb_strlen($text) > 100) {
        $truncated_source .= '...';
      }
      $this->log('notice', 'Translation source (first 100 chars): @source', [
        '@source' => $truncated_source,
      ]);
    } else {
      $this->log('notice', 'Starting translation of @length characters to @language', [
        '@length' => mb_strlen($text),
        '@language' => $target_language,
      ]);
    }
    
    // Log the configuration parameters being used
    $this->log('info', 'Translation request settings: model=@model, temperature=@temp, timeout=@timeout', [
      '@model' => $model,
      '@temp' => $temperature,
      '@timeout' => $timeout,
    ]);
    
    // Add language instruction to the system prompt.
    $full_system_prompt = $system_prompt . ' Translate it to ' . $target_language . '.';
    
    try {
      $startTime = microtime(TRUE);
      
      $response = $this->httpClient->post('https://api.openai.com/v1/chat/completions', [
        'headers' => [
          'Authorization' => 'Bearer ' . $api_key,
          'Content-Type' => 'application/json',
        ],
        'json' => [
          'model' => $model,
          'messages' => [
            [
              'role' => 'system',
              'content' => $full_system_prompt,
            ],
            [
              'role' => 'user',
              'content' => $text,
            ],
          ],
          'temperature' => (float) $temperature,
        ],
        // Use configured timeout settings
        'timeout' => (int) $timeout,
        'connect_timeout' => (int) $connect_timeout,
        'http_errors' => false,     // Don't throw exceptions for HTTP errors
      ]);
      
      $endTime = microtime(TRUE);
      $duration = round($endTime - $startTime, 2);
      
      $this->log('info', 'OpenAI API request completed in @seconds seconds', [
        '@seconds' => $duration,
      ]);
      
      $result = json_decode((string) $response->getBody(), TRUE);
      
      // Check for HTTP error status codes
      $statusCode = $response->getStatusCode();
      if ($statusCode >= 400) {
        $error_message = isset($result['error']['message']) ? $result['error']['message'] : "HTTP error {$statusCode}";
        $this->log('error', 'OpenAI API returned HTTP @status: @message', [
          '@status' => $statusCode, 
          '@message' => $error_message,
        ]);
        throw new \Exception($this->t('Error from OpenAI API: @error', ['@error' => $error_message]));
      }
      
      if (isset($result['choices'][0]['message']['content'])) {
        $translated_text = $result['choices'][0]['message']['content'];
        
        // Log the first 100 characters of the translated text
        if ($show_sample_text) {
          $truncated_translation = mb_substr($translated_text, 0, 100);
          if (mb_strlen($translated_text) > 100) {
            $truncated_translation .= '...';
          }
          $this->log('notice', 'Translation result (first 100 chars): @translation', [
            '@translation' => $truncated_translation,
          ]);
        } else {
          $this->log('notice', 'Completed translation of @length characters to @language', [
            '@length' => mb_strlen($translated_text),
            '@language' => $target_language,
          ]);
        }
        
        return $translated_text;
      }
      else {
        throw new \Exception($this->t('Unexpected API response structure.'));
      }
    }
    catch (RequestException $e) {
      $this->log('error', 'OpenAI API error: @error', ['@error' => $e->getMessage()]);
      throw new \Exception($this->t('Error calling OpenAI API: @error', ['@error' => $e->getMessage()]));
    }
  }
}