<?php

namespace Drupal\ai_content_translation\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\ai_content_translation\Service\OpenAITranslationService;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;

/**
 * Controller for AI translation functionality.
 */
class AITranslationController extends ControllerBase {
  use DependencySerializationTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The OpenAI translation service.
   *
   * @var \Drupal\ai_content_translation\Service\OpenAITranslationService
   */
  protected $translationService;

  /**
   * The language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a new AITranslationController object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\ai_content_translation\Service\OpenAITranslationService $translation_service
   *   The OpenAI translation service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $language_manager
   *   The language manager.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    OpenAITranslationService $translation_service,
    LanguageManagerInterface $language_manager,
    MessengerInterface $messenger,
    ConfigFactoryInterface $config_factory
  ) {
    $this->entityTypeManager = $entity_type_manager;
    $this->translationService = $translation_service;
    $this->languageManager = $language_manager;
    $this->messenger = $messenger;
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('ai_content_translation.openai_translation'),
      $container->get('language_manager'),
      $container->get('messenger'),
      $container->get('config.factory')
    );
  }
  
  /**
   * Logs a message respecting the configured log settings.
   *
   * @param string $level
   *   The log level ('error', 'notice', 'info')
   * @param string $message
   *   The message to log.
   * @param array $context
   *   The context parameters.
   */
  protected function log($level, $message, array $context = []) {
    $config = $this->configFactory->get('ai_content_translation.settings');
    $enable_logging = $config->get('enable_logging') ?? TRUE;
    $configured_level = $config->get('log_level') ?: 'notice';
    
    // Always log errors regardless of settings
    if ($level === 'error') {
      \Drupal::logger('ai_content_translation')->error($message, $context);
      return;
    }
    
    // Skip if logging is disabled
    if (!$enable_logging) {
      return;
    }
    
    // Check if the current log level should be logged based on config
    $should_log = FALSE;
    switch ($configured_level) {
      case 'error':
        // Only errors, which we already handled above
        $should_log = FALSE;
        break;
      
      case 'notice':
        // Errors and notices
        $should_log = ($level === 'notice');
        break;
        
      case 'info':
        // All levels
        $should_log = TRUE;
        break;
    }
    
    if ($should_log) {
      \Drupal::logger('ai_content_translation')->$level($message, $context);
    }
  }
  
  /**
   * Checks if a value should be translated.
   *
   * @param mixed $value
   *   The value to check.
   *
   * @return bool
   *   TRUE if the value should be translated, FALSE otherwise.
   */
  protected function shouldTranslate($value) {
    // Skip NULL values
    if ($value === NULL) {
      return FALSE;
    }
    
    // Skip empty strings or strings that only contain whitespace
    if (is_string($value) && trim($value) === '') {
      return FALSE;
    }
    
    return TRUE;
  }

  /**
   * Translates image field alt and title attributes.
   *
   * @param object $field
   *   The image field to process.
   * @param \Drupal\Core\Entity\ContentEntityInterface $translation
   *   The entity translation.
   * @param string $field_name
   *   The name of the field.
   * @param string $language_name
   *   The human-readable target language name.
   */
  protected function translateImageAttributes($field, ContentEntityInterface $translation, $field_name, $language_name) {
    $this->log('notice', 'Translating image attributes for field: @field_name', [
      '@field_name' => $field_name,
    ]);
    
    foreach ($field as $delta => $item) {
      $updated_item = [
        'target_id' => $item->target_id,
        'width' => $item->width ?? NULL,
        'height' => $item->height ?? NULL,
      ];
      
      // Translate alt text if it exists
      if (isset($item->alt) && $this->shouldTranslate($item->alt)) {
        $this->log('info', 'Translating ALT text for image in field: @field_name', [
          '@field_name' => $field_name,
        ]);
        $translated_alt = $this->translationService->translateText($item->alt, $language_name);
        $updated_item['alt'] = $translated_alt;
      } else {
        $updated_item['alt'] = $item->alt ?? '';
      }
      
      // Translate title text if it exists
      if (isset($item->title) && $this->shouldTranslate($item->title)) {
        $this->log('info', 'Translating title text for image in field: @field_name', [
          '@field_name' => $field_name,
        ]);
        $translated_title = $this->translationService->translateText($item->title, $language_name);
        $updated_item['title'] = $translated_title;
      } else {
        $updated_item['title'] = $item->title ?? '';
      }
      
      // Keep other attributes if they exist
      if (isset($item->display)) {
        $updated_item['display'] = $item->display;
      }
      
      // Set the updated values back to the translation
      $translation->get($field_name)->set($delta, $updated_item);
    }
  }

  /**
   * Recursively translates all content entities including paragraphs.
   *
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to translate.
   * @param string $target_lang
   *   The target language code.
   * @param string $language_name
   *   The human-readable language name.
   * @param array $processed_entities
   *   Array of already processed entity IDs to prevent infinite recursion.
   * @param \Drupal\Core\Entity\ContentEntityInterface $root_entity
   *   The root entity (node) being translated, used to verify ownership.
   * 
   * @return array
   *   An array of entity IDs that were translated or need to be translated.
   */
  protected function translateContentEntities(ContentEntityInterface $entity, $target_lang, $language_name, array &$processed_entities = [], ContentEntityInterface $root_entity = NULL) {
    // Set root entity to the current entity if not provided (first call)
    if ($root_entity === NULL) {
      $root_entity = $entity;
    }
    
    // Generate a unique ID for this entity to prevent recursion
    $entity_id = $entity->getEntityTypeId() . ':' . $entity->id();
    
    // Log the entity being processed
    $this->log('notice', 'Processing entity: @type #@id', [
      '@type' => $entity->getEntityTypeId(),
      '@id' => $entity->id(),
    ]);
    
    // Skip if already processed
    if (in_array($entity_id, $processed_entities)) {
      $this->log('info', 'Skipping already processed entity: @entity_id', [
        '@entity_id' => $entity_id,
      ]);
      return [];
    }
    
    // Skip file and media entities completely
    if (in_array($entity->getEntityTypeId(), ['file', 'media'])) {
      $this->log('info', 'Skipping file/media entity: @type #@id', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
      ]);
      return [];
    }
    
    // Mark this entity as processed
    $processed_entities[] = $entity_id;
    
    // Collect entities to translate
    $entities_to_translate = [$entity_id => [
      'entity_type' => $entity->getEntityTypeId(),
      'entity_id' => $entity->id(),
      'target_lang' => $target_lang,
      'language_name' => $language_name,
      'root_entity_type' => $root_entity->getEntityTypeId(),
      'root_entity_id' => $root_entity->id(),
    ]];
    
    // Get all fields to find entity references
    foreach ($entity->getFields() as $field_name => $field) {
      // Skip internal fields and non-entity reference fields
      if (in_array($field_name, ['langcode', 'default_langcode', 'content_translation_source', 
          'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
          'content_translation_changed', 'revision_uid', 'revision_timestamp', 'revision_log'])) {
        continue;
      }
      
      $field_definition = $field->getFieldDefinition();
      $field_type = $field_definition->getType();
      
      // Handle entity reference fields (including paragraphs and other content entities)
      if ($field_type == 'entity_reference' || $field_type == 'entity_reference_revisions') {
        $target_type = $field_definition->getSetting('target_type');
        
        $this->log('info', 'Found reference field: @field_name, target type: @target_type', [
          '@field_name' => $field_name,
          '@target_type' => $target_type,
        ]);
        
        // Skip file and media field references
        if (in_array($target_type, ['file', 'media'])) {
          $this->log('info', 'Skipping file/media reference field: @field_name', [
            '@field_name' => $field_name,
          ]);
          continue;
        }
        
        // Process all referenced entities
        foreach ($field as $delta => $item) {
          if ($item->entity && $item->entity instanceof ContentEntityInterface) {
            $referenced_entity = $item->entity;
            
            $this->log('info', 'Found referenced entity: @type #@id in field @field_name', [
              '@type' => $referenced_entity->getEntityTypeId(),
              '@id' => $referenced_entity->id(),
              '@field_name' => $field_name,
            ]);
            
            // Skip file and media entities
            if (in_array($referenced_entity->getEntityTypeId(), ['file', 'media'])) {
              $this->log('info', 'Skipping file/media referenced entity: @type #@id', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
              ]);
              continue;
            }
            
            // Verify this entity belongs to our node by checking the parent entity
            $belongs_to_current_node = FALSE;
            
            // For paragraphs, check if parent entity matches our root node
            if ($referenced_entity->getEntityTypeId() == 'paragraph') {
              // If paragraph has a parent_field_name and parent_type property, check them
              if (method_exists($referenced_entity, 'getParentEntity')) {
                $parent = $referenced_entity->getParentEntity();
                if ($parent) {
                  // Either the parent is our root entity or a descendant
                  if ($parent->id() == $root_entity->id() && $parent->getEntityTypeId() == $root_entity->getEntityTypeId()) {
                    $belongs_to_current_node = TRUE;
                    $this->log('info', 'Paragraph @id belongs to root entity directly', [
                      '@id' => $referenced_entity->id(),
                    ]);
                  }
                  // Or the parent could be another paragraph that belongs to our root entity
                  else if ($parent->getEntityTypeId() == 'paragraph') {
                    // We're already processing this, so it must belong to our structure
                    $belongs_to_current_node = TRUE;
                    $this->log('info', 'Paragraph @id belongs to root entity via another paragraph', [
                      '@id' => $referenced_entity->id(),
                    ]);
                  }
                }
              } else {
                // Fallback for older Paragraph module versions
                // Since we're recursively following references from the root node,
                // any paragraph we find must be part of the current node's structure
                $belongs_to_current_node = TRUE;
                $this->log('info', 'Paragraph @id belongs to structure (fallback method)', [
                  '@id' => $referenced_entity->id(),
                ]);
              }
            } else {
              // For non-paragraph entities, we trust the reference structure
              // If we found it by traversing from the root node, it belongs to the node
              $belongs_to_current_node = TRUE;
              $this->log('info', 'Non-paragraph entity @type #@id belongs to structure', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
              ]);
            }
            
            // Only process if this entity belongs to our node
            if ($belongs_to_current_node && $referenced_entity->isTranslatable()) {
              $this->log('notice', 'Entity @type #@id will be translated to @lang', [
                '@type' => $referenced_entity->getEntityTypeId(),
                '@id' => $referenced_entity->id(),
                '@lang' => $target_lang,
              ]);
              
              // Add to list of entities to translate
              $referenced_entity_id = $referenced_entity->getEntityTypeId() . ':' . $referenced_entity->id();
              $entities_to_translate[$referenced_entity_id] = [
                'entity_type' => $referenced_entity->getEntityTypeId(),
                'entity_id' => $referenced_entity->id(),
                'target_lang' => $target_lang,
                'language_name' => $language_name,
                'root_entity_type' => $root_entity->getEntityTypeId(),
                'root_entity_id' => $root_entity->id(),
              ];
              
              // Recursively find more entities to translate
              $child_entities = $this->translateContentEntities($referenced_entity, $target_lang, $language_name, $processed_entities, $root_entity);
              $entities_to_translate = array_merge($entities_to_translate, $child_entities);
            } else {
              if (!$belongs_to_current_node) {
                $this->log('info', 'Entity @type #@id does not belong to the current node structure', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                ]);
              }
              if (!$referenced_entity->isTranslatable()) {
                $this->log('info', 'Entity @type #@id is not translatable', [
                  '@type' => $referenced_entity->getEntityTypeId(),
                  '@id' => $referenced_entity->id(),
                ]);
              }
            }
          }
        }
      }
    }
    
    $this->log('notice', 'Completed processing entity: @type #@id', [
      '@type' => $entity->getEntityTypeId(),
      '@id' => $entity->id(),
    ]);
    
    return $entities_to_translate;
  }

  /**
   * Batch processing callback: Translates a single entity.
   *
   * @param array $entity_info
   *   Entity information array.
   * @param array $context
   *   Batch context array.
   */
  public static function batchTranslateEntity($entity_info, &$context) {
    // Load the entity.
    $entity_type = $entity_info['entity_type'];
    $entity_id = $entity_info['entity_id'];
    $target_lang = $entity_info['target_lang'];
    $language_name = $entity_info['language_name'];
    
    $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
    
    if (!$entity) {
      $context['results']['errors'][] = t('Entity @type #@id not found.', [
        '@type' => $entity_type,
        '@id' => $entity_id,
      ]);
      return;
    }
    
    if (!$entity->isTranslatable()) {
      $context['results']['skipped'][] = t('Entity @type #@id is not translatable.', [
        '@type' => $entity_type,
        '@id' => $entity_id,
      ]);
      return;
    }
    
    try {
      // Get entity's source language - make sure we have a valid language code, not 'und'
      $source_langcode = $entity->language()->getId();
      if ($source_langcode === 'und') {
        // If for some reason the entity has 'und' as its language, use site default
        $source_langcode = \Drupal::languageManager()->getDefaultLanguage()->getId();
      }
      
      // Create a translation if it doesn't exist.
      if (!$entity->hasTranslation($target_lang)) {
        // For paragraphs, we need to carefully preserve parent references
        if ($entity_type === 'paragraph') {
          // Create a translation with only essential fields preserved
          $translation_values = [
            'langcode' => $target_lang,
            'status' => $entity->get('status')->value,
            // Set proper translation metadata
            'content_translation_source' => $source_langcode,
            'content_translation_outdated' => 0,
          ];
          
          // Preserve these important paragraph references from the source language
          // and prevent them from being translated to avoid database errors
          if ($entity->hasField('parent_id')) {
            $translation_values['parent_id'] = $entity->get('parent_id')->getValue();
          }
          if ($entity->hasField('parent_type')) {
            $translation_values['parent_type'] = $entity->get('parent_type')->getValue();
          }
          if ($entity->hasField('parent_field_name')) {
            $translation_values['parent_field_name'] = $entity->get('parent_field_name')->getValue();
          }
          if ($entity->hasField('behavior_settings')) {
            $translation_values['behavior_settings'] = $entity->get('behavior_settings')->getValue();
          }
          
          $translation = $entity->addTranslation($target_lang, $translation_values);
        }
        else {
          // For non-paragraph entities, normal translation is fine
          // but still ensure correct translation source
          $values = $entity->toArray();
          $values['content_translation_source'] = $source_langcode;
          $values['content_translation_outdated'] = 0;
          
          $translation = $entity->addTranslation($target_lang, $values);
        }
      }
      else {
        $translation = $entity->getTranslation($target_lang);
      }
      
      // Get OpenAI translation service
      $translation_service = \Drupal::service('ai_content_translation.openai_translation');
      $config = \Drupal::config('ai_content_translation.settings');
      $enable_logging = $config->get('enable_logging') ?? TRUE;
      $log_level = $config->get('log_level') ?: 'notice';
      
      // Helper function to log messages
      $log = function($level, $message, $context = []) use ($enable_logging, $log_level) {
        // Always log errors regardless of settings
        if ($level === 'error') {
          \Drupal::logger('ai_content_translation')->error($message, $context);
          return;
        }
        
        // Skip if logging is disabled
        if (!$enable_logging) {
          return;
        }
        
        // Check if the current log level should be logged based on config
        $should_log = FALSE;
        switch ($log_level) {
          case 'error':
            // Only errors, which we already handled above
            $should_log = FALSE;
            break;
          
          case 'notice':
            // Errors and notices
            $should_log = ($level === 'notice');
            break;
            
          case 'info':
            // All levels
            $should_log = TRUE;
            break;
        }
        
        if ($should_log) {
          \Drupal::logger('ai_content_translation')->$level($message, $context);
        }
      };
      
      // Helper function to check if value should be translated
      $shouldTranslate = function($value) {
        // Skip NULL values
        if ($value === NULL) {
          return FALSE;
        }
        
        // Skip empty strings or strings that only contain whitespace
        if (is_string($value) && trim($value) === '') {
          return FALSE;
        }
        
        return TRUE;
      };
      
      // Translate title if it exists (title is a text field).
      if ($entity->hasField('title')) {
        $title_value = $entity->get('title')->value;
        if ($shouldTranslate($title_value)) {
          $translated_title = $translation_service->translateText($title_value, $language_name);
          $translation->set('title', $translated_title);
          $log('info', 'Translated title field for @type #@id', [
            '@type' => $entity->getEntityTypeId(),
            '@id' => $entity->id(),
          ]);
        }
      }

      // Translate body if it exists (body is a text field).
      if ($entity->hasField('body')) {
        $body_value = $entity->get('body')->value;
        $body_format = $entity->get('body')->format;
        
        if ($shouldTranslate($body_value)) {
          $translated_body = $translation_service->translateText($body_value, $language_name);
          $translation->set('body', [
            'value' => $translated_body,
            'format' => $body_format,
          ]);
          $log('info', 'Translated body field for @type #@id', [
            '@type' => $entity->getEntityTypeId(),
            '@id' => $entity->id(),
          ]);
        }
      }

      // Translate all text fields that might contain content.
      foreach ($entity->getFields() as $field_name => $field) {
        // Skip internal fields, reference fields, and already translated fields
        if (in_array($field_name, [
          'title', 'body', 'langcode', 'default_langcode', 'content_translation_source', 
          'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
          'content_translation_changed', 'parent_id', 'parent_type', 'parent_field_name',
          'behavior_settings'
        ])) {
          continue;
        }

        $field_type = $field->getFieldDefinition()->getType();
        
        // Handle image fields to translate alt and title text
        if ($field_type === 'image') {
          $log('info', 'Processing image field for alt/title translation: @field_name', [
            '@field_name' => $field_name,
          ]);
          
          foreach ($field as $delta => $item) {
            $updated_item = [
              'target_id' => $item->target_id,
              'width' => $item->width ?? NULL,
              'height' => $item->height ?? NULL,
            ];
            
            // Translate alt text if it exists.
            if (isset($item->alt) && $shouldTranslate($item->alt)) {
              $translated_alt = $translation_service->translateText($item->alt, $language_name);
              $updated_item['alt'] = $translated_alt;
            }
            else {
              $updated_item['alt'] = $item->alt ?? '';
            }
            
            // Translate title text if it exists.
            if (isset($item->title) && $shouldTranslate($item->title)) {
              $translated_title = $translation_service->translateText($item->title, $language_name);
              $updated_item['title'] = $translated_title;
            }
            else {
              $updated_item['title'] = $item->title ?? '';
            }
            
            // Keep other attributes if they exist.
            if (isset($item->display)) {
              $updated_item['display'] = $item->display;
            }
            
            // Set the updated values back to the translation.
            $translation->get($field_name)->set($delta, $updated_item);
          }
          
          continue;
        }
        
        // Skip non-text fields
        if (!in_array($field_type, ['text', 'text_long', 'text_with_summary', 'string', 'string_long'])) {
          continue;
        }

        // Handle text fields only
        $log('info', 'Translating text field on entity: @field_name', [
          '@field_name' => $field_name,
        ]);
        
        foreach ($field as $delta => $item) {
          if ($shouldTranslate($item->value)) {
            $translated_value = $translation_service->translateText($item->value, $language_name);
            $translation->get($field_name)->set($delta, ['value' => $translated_value, 'format' => $item->format ?? NULL]);
          }
        }
      }
      
      // Make sure translation source is properly set before saving
      if ($translation->hasField('content_translation_source')) {
        // Ensure we're using a valid language code, not 'und'
        $current_source = $translation->get('content_translation_source')->value;
        if ($current_source === 'und' || empty($current_source)) {
          $translation->set('content_translation_source', $source_langcode);
        }
      }
      
      // Save the entity with the new translation.
      $translation->save();
      
      $log('notice', 'Completed text field translation for @type #@id', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
      ]);
      
      // Record success.
      if (!isset($context['results']['translated'])) {
        $context['results']['translated'] = [];
      }
      $context['results']['translated'][] = $entity_type . ':' . $entity_id;
      
      // Update progress message.
      $context['message'] = t('Translated @type #@id', [
        '@type' => $entity_type,
        '@id' => $entity_id,
      ]);
    }
    catch (\Exception $e) {
      if (!isset($context['results']['errors'])) {
        $context['results']['errors'] = [];
      }
      $context['results']['errors'][] = t('Error translating @type #@id: @error', [
        '@type' => $entity_type,
        '@id' => $entity_id,
        '@error' => $e->getMessage(),
      ]);
      
      // Log the error
      \Drupal::logger('ai_content_translation')->error('Error in batch translation for @type #@id: @error', [
        '@type' => $entity_type,
        '@id' => $entity_id,
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Batch finished callback.
   *
   * @param bool $success
   *   Indicates whether the batch process was successful.
   * @param array $results
   *   Results information passed from the processing callback.
   * @param array $operations
   *   A list of the operations that were processed.
   */
  public static function batchFinished($success, $results, $operations) {
    $messenger = \Drupal::messenger();
    
    if ($success) {
      // Count translated entities.
      $translated_count = isset($results['translated']) ? count($results['translated']) : 0;
      
      if ($translated_count > 0) {
        $messenger->addStatus(t('Successfully translated @count entities.', [
          '@count' => $translated_count,
        ]));
      }
      
      // Report errors.
      if (!empty($results['errors'])) {
        foreach ($results['errors'] as $error) {
          $messenger->addError($error);
        }
      }
      
      // Report skipped entities.
      if (!empty($results['skipped'])) {
        foreach ($results['skipped'] as $skipped) {
          $messenger->addWarning($skipped);
        }
      }
      
      // Get the primary entity from the batch data.
      if (!empty($results['primary_entity'])) {
        $entity_type = $results['primary_entity']['entity_type'];
        $entity_id = $results['primary_entity']['entity_id'];
        $target_lang = $results['primary_entity']['target_lang'];
        
        // Load primary entity to generate links.
        $entity = \Drupal::entityTypeManager()->getStorage($entity_type)->load($entity_id);
        
        if ($entity && $entity->hasTranslation($target_lang)) {
          $language = \Drupal::languageManager()->getLanguage($target_lang);
          
          // Create links for the success message.
          $view_url = $entity->toUrl('canonical', ['language' => $language]);
          $view_link = Link::fromTextAndUrl(t('View translation'), $view_url)->toString();

          $edit_url = $entity->toUrl('edit-form', ['language' => $language]);
          $edit_link = Link::fromTextAndUrl(t('Edit translation'), $edit_url)->toString();

          // Get entity title for the success message.
          $entity_title = '';
          if ($entity->hasField('title') && !empty($entity->get('title')->value)) {
            $entity_title = $entity->get('title')->value;
          }
          elseif ($entity->hasField('name') && !empty($entity->get('name')->value)) {
            $entity_title = $entity->get('name')->value;
          }
          elseif ($entity->hasField('label') && !empty($entity->get('label')->value)) {
            $entity_title = $entity->get('label')->value;
          }

          // Add success message with title and links.
          if (!empty($entity_title)) {
            $messenger->addStatus(t('AI translation of "@title" to @language has been generated. @view_link | @edit_link', [
              '@title' => $entity_title,
              '@language' => $language->getName(),
              '@view_link' => $view_link,
              '@edit_link' => $edit_link,
            ]));
          } else {
            $messenger->addStatus(t('AI translation to @language has been generated. @view_link | @edit_link', [
              '@language' => $language->getName(),
              '@view_link' => $view_link,
              '@edit_link' => $edit_link,
            ]));
          }
        }
      }
    }
    else {
      $messenger->addError(t('An error occurred during the translation process.'));
    }
    
    // Check if we have a return URL to redirect to
    if (!empty($results['return_url'])) {
      // Set a session variable for redirect_after_batch
      $_SESSION['ai_translation_redirect_after_batch'] = $results['return_url'];
    }
  }
  
  /**
   * Redirects the user after a batch process completes.
   */
  public function batchRedirectPage() {
    // Get redirect URL from session
    $redirect_url = isset($_SESSION['ai_translation_redirect_after_batch']) 
      ? $_SESSION['ai_translation_redirect_after_batch'] 
      : '/admin/content';
    
    // Clear the session variable
    unset($_SESSION['ai_translation_redirect_after_batch']);
    
    // Return a redirect response
    return new RedirectResponse($redirect_url);
  }

  /**
   * Translates an entity using AI.
   *
   * @param string $entity_type
   *   The entity type ID.
   * @param mixed $entity_id
   *   The entity.
   * @param string $target_lang
   *   The target language code.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse
   *   A redirect response.
   */
  public function translate($entity_type, $entity_id, $target_lang) {
    // Get the entity.
    if (is_object($entity_id)) {
      $entity = $entity_id;
    }
    else {
      $entity = $this->entityTypeManager->getStorage($entity_type)->load($entity_id);
    }

    if (!$entity) {
      $this->messenger->addError($this->t('Entity not found.'));
      return $this->redirect('<front>');
    }

    if (!$entity instanceof ContentEntityInterface || !$entity->isTranslatable()) {
      $this->messenger->addError($this->t('Entity is not translatable.'));
      return $this->redirect('<front>');
    }

    // Get the language object.
    $language = $this->languageManager->getLanguage($target_lang);
    if (!$language) {
      $this->messenger->addError($this->t('Invalid target language.'));
      return $this->redirect('<front>');
    }

    // Check if translation already exists.
    if ($entity->hasTranslation($target_lang)) {
      $this->messenger->addWarning($this->t('Translation already exists for @language.', [
        '@language' => $language->getName(),
      ]));
      return $this->redirect('entity.' . $entity_type . '.content_translation_overview', [
        $entity_type => $entity->id(),
      ]);
    }

    // Get the referring URL for redirecting back after translation
    $request = \Drupal::request();
    $referer = $request->headers->get('referer');
    
    // If no referer, default to content admin page
    if (empty($referer)) {
      $referer = Url::fromRoute('system.admin_content')->toString();
    }

    try {
      $this->log('notice', 'Starting AI translation of @type #@id to @language', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@language' => $language->getName(),
      ]);
      
      // Check if batch processing is enabled.
      $config = $this->configFactory->get('ai_content_translation.settings');
      $use_batch_processing = $config->get('use_batch_processing') ?? FALSE;
      
      if ($use_batch_processing) {
        // Collect all entities that need to be translated, but don't translate them yet.
        $processed_entities = [];
        $entities_to_translate = $this->translateContentEntities($entity, $target_lang, $language->getName(), $processed_entities, $entity);
        
        // Set up the batch process.
        $batch_builder = new BatchBuilder();
        $batch_builder->setTitle($this->t('Translating to @language', ['@language' => $language->getName()]));
        $batch_builder->setInitMessage($this->t('Starting translation process...'));
        $batch_builder->setProgressMessage($this->t('Translated @current out of @total entities.'));
        $batch_builder->setErrorMessage($this->t('An error occurred during translation.'));
        
        // Add the primary entity to the batch for later reference.
        $batch_builder->addOperation('Drupal\ai_content_translation\Controller\AITranslationController::batchTranslateEntity', [
          [
            'entity_type' => $entity->getEntityTypeId(),
            'entity_id' => $entity->id(),
            'target_lang' => $target_lang,
            'language_name' => $language->getName(),
            'root_entity_type' => $entity->getEntityTypeId(),
            'root_entity_id' => $entity->id(),
          ]
        ]);
        
        // Store primary entity info for the finished callback.
        $batch_builder->setFinishCallback('Drupal\ai_content_translation\Controller\AITranslationController::batchFinished');
        
        // Add operations for each entity that needs to be translated.
        $batch_size = $config->get('batch_size') ?: 5;
        $chunks = array_chunk($entities_to_translate, $batch_size, TRUE);
        
        foreach ($chunks as $chunk) {
          foreach ($chunk as $entity_info) {
            // Skip the primary entity which we've already added.
            if ($entity_info['entity_type'] == $entity->getEntityTypeId() && 
                $entity_info['entity_id'] == $entity->id()) {
              continue;
            }
            
            // Use the static class method directly to avoid serialization issues
            $batch_builder->addOperation('Drupal\ai_content_translation\Controller\AITranslationController::batchTranslateEntity', [$entity_info]);
          }
        }
        
        // Set the batch.
        $batch = $batch_builder->toArray();
        $batch['progressive'] = TRUE;
        $batch['results']['primary_entity'] = [
          'entity_type' => $entity->getEntityTypeId(),
          'entity_id' => $entity->id(),
          'target_lang' => $target_lang,
        ];
        // Store the referrer URL in the batch results
        $batch['results']['return_url'] = $referer;
        
        batch_set($batch);
        
        // Process the batch and redirect.
        if (PHP_SAPI === 'cli') {
          drush_backend_batch_process();
          return new RedirectResponse($referer);
        }
        else {
          // Redirect to our custom endpoint after batch completion
          return batch_process(Url::fromRoute('ai_content_translation.batch_complete'));
        }
      }
      else {
        // Non-batch processing - original direct translation method.
        // Create a new translation.
        $translation = $entity->addTranslation($target_lang, $entity->toArray());
        
        // Translate title if it exists (title is a text field).
        if ($entity->hasField('title')) {
          $title_value = $entity->get('title')->value;
          if ($this->shouldTranslate($title_value)) {
            $translated_title = $this->translationService->translateText($title_value, $language->getName());
            $translation->set('title', $translated_title);
            $this->log('info', 'Translated title field');
          } else {
            $this->log('info', 'Skipping empty title field');
          }
        }

        // Translate body if it exists (body is a text field).
        if ($entity->hasField('body')) {
          $body_value = $entity->get('body')->value;
          $body_format = $entity->get('body')->format;
          
          if ($this->shouldTranslate($body_value)) {
            $translated_body = $this->translationService->translateText($body_value, $language->getName());
            $translation->set('body', [
              'value' => $translated_body,
              'format' => $body_format,
            ]);
            $this->log('info', 'Translated body field');
          } else {
            $this->log('info', 'Skipping empty body field');
          }
        }

        // Translate all text fields that might contain content.
        foreach ($entity->getFields() as $field_name => $field) {
          // Skip already translated fields
          if (in_array($field_name, ['title', 'body', 'langcode', 'default_langcode', 'content_translation_source', 
              'content_translation_outdated', 'content_translation_uid', 'content_translation_created', 
              'content_translation_changed'])) {
            continue;
          }

          $field_type = $field->getFieldDefinition()->getType();
          
          // Handle image fields to translate alt and title text
          if ($field_type === 'image') {
            $this->log('info', 'Processing image field for alt/title translation: @field_name', [
              '@field_name' => $field_name,
            ]);
            $this->translateImageAttributes($field, $translation, $field_name, $language->getName());
            continue;
          }
          
          // Skip non-text fields
          if (!in_array($field_type, ['text', 'text_long', 'text_with_summary', 'string', 'string_long'])) {
            $this->log('info', 'Skipping non-text field on main entity: @field_name (type: @type)', [
              '@field_name' => $field_name,
              '@type' => $field_type,
            ]);
            continue;
          }

          // Handle text fields only
          $this->log('info', 'Translating text field on main entity: @field_name', [
            '@field_name' => $field_name,
          ]);
          
          foreach ($field as $delta => $item) {
            if ($this->shouldTranslate($item->value)) {
              $translated_value = $this->translationService->translateText($item->value, $language->getName());
              $translation->get($field_name)->set($delta, ['value' => $translated_value, 'format' => $item->format ?? NULL]);
            } else {
              $this->log('info', 'Skipping empty value in field: @field_name', [
                '@field_name' => $field_name,
              ]);
            }
          }
        }

        // Initialize the array for tracking processed entities
        $processed_entities = [];
        
        // Recursively process and translate all paragraph and content entities
        // Pass the entity as the root entity to ensure we only translate entities belonging to this node
        $this->translateContentEntities($entity, $target_lang, $language->getName(), $processed_entities, $entity);

        // Save the entity with the new translation.
        $translation->save();
        
        $this->log('notice', 'Completed AI translation of @type #@id to @language', [
          '@type' => $entity->getEntityTypeId(),
          '@id' => $entity->id(),
          '@language' => $language->getName(),
        ]);

        // Get entity title for the success message
        $entity_title = '';
        if ($entity->hasField('title') && !empty($entity->get('title')->value)) {
          $entity_title = $entity->get('title')->value;
        }
        elseif ($entity->hasField('name') && !empty($entity->get('name')->value)) {
          $entity_title = $entity->get('name')->value;
        }
        elseif ($entity->hasField('label') && !empty($entity->get('label')->value)) {
          $entity_title = $entity->get('label')->value;
        }

        // Create links for the success message
        $view_url = $entity->toUrl('canonical', ['language' => $language]);
        $view_link = Link::fromTextAndUrl($this->t('View translation'), $view_url)->toString();

        $edit_url = $entity->toUrl('edit-form', ['language' => $language]);
        $edit_link = Link::fromTextAndUrl($this->t('Edit translation'), $edit_url)->toString();

        // Add success message with title and links
        if (!empty($entity_title)) {
          $this->messenger->addStatus($this->t('AI translation of "@title" to @language has been generated. @view_link | @edit_link', [
            '@title' => $entity_title,
            '@language' => $language->getName(),
            '@view_link' => $view_link,
            '@edit_link' => $edit_link,
          ]));
        } else {
          $this->messenger->addStatus($this->t('AI translation to @language has been generated. !view_link | !edit_link', [
            '@language' => $language->getName(),
            '@view_link' => $view_link,
            '@edit_link' => $edit_link,
          ]));
        }
        
        // Return to the referring page instead of the edit form
        return new RedirectResponse($referer);
      }
    }
    catch (\Exception $e) {
      $this->log('error', 'Error generating AI translation for @type #@id: @error', [
        '@type' => $entity->getEntityTypeId(),
        '@id' => $entity->id(),
        '@error' => $e->getMessage(),
      ]);
      
      $this->messenger->addError($this->t('Error generating AI translation: @error', [
        '@error' => $e->getMessage(),
      ]));
      
      return $this->redirect('entity.' . $entity_type . '.content_translation_overview', [
        $entity_type => $entity->id(),
      ]);
    }
  }
}