<?php

declare(strict_types=1);

namespace Drupal\ai_eca_agents\Command;

use Drupal\Component\Serialization\Json;
use Drupal\ai_eca_agents\Services\DataProvider\DataProviderInterface;
use Drupal\ai_eca_agents\Services\DataProvider\DataViewModeEnum;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;

/**
 * A console command to debug the data provider.
 */
#[AsCommand(
  name: 'ai-agents-eca:debug:data',
  description: 'Debug the data provider',
)]
final class DebugDataProviderCommand extends Command {

  /**
   * Constructs an DebugCommand object.
   */
  public function __construct(
    protected readonly DataProviderInterface $dataProvider,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  protected function configure(): void {
    $this
      ->addOption('filter', NULL, InputOption::VALUE_OPTIONAL, 'The filter to apply to the provider. Options are "components" or "models"')
      ->addOption('vm', NULL, InputOption::VALUE_OPTIONAL, 'The view mode to use. Options are teaser or full.', DataViewModeEnum::Teaser->value)
      ->addUsage('ai-agents-eca:debug:data --vm=full')
      ->setHelp('Debug the data provider');
  }

  /**
   * {@inheritdoc}
   */
  protected function execute(InputInterface $input, OutputInterface $output): int {
    $this->dataProvider->setViewMode(DataViewModeEnum::from($input->getOption('vm')));
    $response = [
      'components' => $this->dataProvider->getComponents(),
      'models' => $this->dataProvider->getModels(),
    ];
    if (!empty($input->getOption('filter'))) {
      $filter = $input->getOption('filter');
      $response = array_filter($response, function (string $key) use ($filter) {
        return $key === $filter;
      }, ARRAY_FILTER_USE_KEY);
    }

    $response = Json::decode(Json::encode($response));

    $output->writeln(print_r($response, TRUE));

    return self::SUCCESS;
  }

}
