<?php

namespace Drupal\ai_eca_agents\Form;

use Drupal\Core\Batch\BatchBuilder;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\Task\Task;
use Drupal\ai_eca_agents\Plugin\AiAgent\Eca;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;

/**
 * A form to propose a question to AI.
 */
class AskAiForm extends FormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'ai_eca_agents_ask_ai';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form['question'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Propose your question'),
      '#required' => TRUE,
    ];

    // Determine the destination for when the batch process is finished.
    $destination = $this->getRequest()->query->get('destination', Url::fromRoute('entity.eca.collection')->toString());
    $form['destination'] = [
      '#type' => 'value',
      '#value' => $destination,
    ];

    // Determine if an existing model is the subject of the prompt.
    $modelId = $this->getRequest()->query->get('model-id');
    $form['model_id'] = [
      '#type' => 'value',
      '#value' => $modelId,
    ];

    $form['actions'] = [
      '#type' => 'actions',
    ];
    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Ask'),
      '#button_type' => 'primary',
    ];
    $form['#theme'] = 'confirm_form';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $batch = new BatchBuilder();
    $batch->setTitle($this->t('Solving question.'));
    $batch->setInitMessage($this->t('Asking AI which task to perform...'));
    $batch->setErrorMessage($this->t('An error occurred during processing.'));
    $batch->setFinishCallback([self::class, 'batchFinished']);

    $batch->addOperation([self::class, 'initProcess'], [$form_state->getValue('destination')]);
    $batch->addOperation([self::class, 'determineTask'], [
      $form_state->getValue('question'),
      $form_state->getValue('model_id'),
    ]);
    $batch->addOperation([self::class, 'executeTask']);

    batch_set($batch->toArray());
  }

  /**
   * Batch operation for initializing the process.
   *
   * @param string $destination
   *   The destination for when the process is finished.
   * @param array $context
   *   The context.
   */
  public static function initProcess(string $destination, array &$context): void {
    $context['message'] = t('Initializing...');

    $context['results']['destination'] = $destination;
  }

  /**
   * Batch operation for determining the task to execute.
   *
   * @param string $question
   *   The question of the user.
   * @param string|null $modelId
   *   The ECA model ID.
   * @param array $context
   *   The context.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public static function determineTask(string $question, ?string $modelId, array &$context): void {
    $context['message'] = t('Task determined. Executing...');

    $context['sandbox']['question'] = $question;
    $context['sandbox']['model_id'] = $modelId;
    $agent = self::initAgent($context);

    // Let the agent decide how it can answer the question.
    $solvability = $agent->determineSolvability();

    if ($solvability === AiAgentInterface::JOB_NOT_SOLVABLE) {
      $context['results']['error'] = t('The AI agent could not solve the task.');

      return;
    }

    // Redirect to the convert-endpoint of the BPMN.io-module.
    $bpmnIoIsAvailable = \Drupal::moduleHandler()->moduleExists('bpmn_io');
    if ($solvability === AiAgentInterface::JOB_SOLVABLE && $bpmnIoIsAvailable && !empty($modelId)) {
      $context['results']['destination'] = Url::fromRoute('bpmn_io.convert', [
        'eca' => $modelId,
      ])->toString();
    }

    $context['results']['dto'] = $agent->getDto();
  }

  /**
   * Batch operation for executing the task.
   *
   * @param array $context
   *   The context.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  public static function executeTask(array &$context): void {
    $context['message'] = t('Task executed. Gathering feedback...');

    $agent = self::initAgent($context);

    // Solve the question.
    $response = $agent->solve();
    $context['results']['response'] = $response;
  }

  /**
   * Callback for when the batch is finished.
   *
   * @param bool $success
   *   A boolean indicating a successful execution.
   * @param array $results
   *   The collection of results.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Returns a http-response.
   */
  public static function batchFinished(bool $success, array $results): Response {
    if (!empty($results['response'])) {
      \Drupal::messenger()->addStatus($results['response']);
    }

    return new RedirectResponse($results['destination']);
  }

  /**
   * Initialize the AI agent.
   *
   * @param array $context
   *   The context.
   *
   * @return \Drupal\ai_eca_agents\Plugin\AiAgent\Eca
   *   Returns the AI agent.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginException
   */
  protected static function initAgent(array $context): Eca {
    // Initiate the agent.
    /** @var \Drupal\ai_agents\PluginInterfaces\AiAgentInterface $agent */
    $agent = \Drupal::service('plugin.manager.ai_agents')
      ->createInstance('eca');
    assert($agent instanceof Eca);

    $dto = [];
    if (!empty($context['sandbox']['model_id'])) {
      $dto['model_id'] = $context['sandbox']['model_id'];
    }
    if (!empty($context['results']['dto']) && is_array($context['results']['dto'])) {
      $dto = array_merge($dto, $context['results']['dto']);
      $dto['setup_agent'] = TRUE;
    }
    if (!empty($dto)) {
      $agent->setDto($dto);
    }

    // Prepare the task.
    $question = $context['sandbox']['question'] ?? NULL;
    $question = $dto['task_description'] ?? $question;
    $task = new Task($question);
    $agent->setTask($task);

    // Set the provider.
    /** @var \Drupal\ai\AiProviderPluginManager $providerManager */
    $providerManager = \Drupal::service('ai.provider');
    $provider = $providerManager->getDefaultProviderForOperationType('chat');
    $agent->setAiProvider($providerManager->createInstance($provider['provider_id']));
    $agent->setModelName($provider['model_id']);
    $agent->setAiConfiguration([]);

    return $agent;
  }

}
