<?php

namespace Drupal\ai_eca_agents\Plugin\AiAgent;

use Drupal\ai_eca_agents\Schema\Eca as EcaSchema;
use Drupal\ai_eca_agents\TypedData\EcaModelDefinition;
use Drupal\Component\Render\MarkupInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_agents\Attribute\AiAgent;
use Drupal\ai_agents\PluginBase\AiAgentBase;
use Drupal\ai_agents\PluginInterfaces\AiAgentInterface;
use Drupal\ai_agents\Task\TaskInterface;
use Drupal\ai_eca_agents\Services\DataProvider\DataProviderInterface;
use Drupal\ai_eca_agents\Services\DataProvider\DataViewModeEnum;
use Drupal\ai_eca_agents\Services\EcaRepository\EcaRepositoryInterface;
use Drupal\eca\Entity\Eca as EcaEntity;
use Illuminate\Support\Arr;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * The ECA agent.
 */
#[AiAgent(
  id: 'eca',
  label: new TranslatableMarkup('ECA Agent'),
)]
class Eca extends AiAgentBase implements ContainerFactoryPluginInterface {

  /**
   * The DTO.
   *
   * @var array
   */
  protected array $dto;

  /**
   * The ECA entity.
   *
   * @var \Drupal\eca\Entity\Eca|null
   */
  protected ?EcaEntity $model = NULL;

  /**
   * The ECA data provider.
   *
   * @var \Drupal\ai_eca_agents\Services\DataProvider\DataProviderInterface
   */
  protected DataProviderInterface $dataProvider;

  /**
   * The ECA helper.
   *
   * @var \Drupal\ai_eca_agents\Services\EcaRepository\EcaRepositoryInterface
   */
  protected EcaRepositoryInterface $ecaRepository;

  /**
   * The serializer.
   *
   * @var \Symfony\Component\Serializer\SerializerInterface
   */
  protected SerializerInterface $serializer;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->dataProvider = $container->get('ai_eca_agents.services.data_provider');
    $instance->ecaRepository = $container->get('ai_eca_agents.services.eca_repository');
    $instance->serializer = $container->get('serializer');
    $instance->dto = [
      'task_description' => '',
      'feedback' => '',
      'questions' => [],
      'data' => [],
      'logs' => [],
    ];

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getId(): string {
    return 'eca';
  }

  /**
   * {@inheritdoc}
   */
  public function agentsNames(): array {
    return ['Event-Condition-Action agent'];
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    return $this->agentHelper->isModuleEnabled('eca');
  }

  /**
   * {@inheritdoc}
   */
  public function getRetries(): int {
    return 2;
  }

  /**
   * {@inheritdoc}
   */
  public function hasAccess() {
    if (!$this->currentUser->hasPermission('administer eca')) {
      return AccessResult::forbidden();
    }

    return parent::hasAccess();
  }

  /**
   * {@inheritdoc}
   */
  public function agentsCapabilities(): array {
    return [
      'eca' => [
        'name' => 'Event-Condition-Action Agent',
        'description' => 'This is agent is capable of adding, editing or informing about Event-Condition-Action models on a Drupal website. Note that it does not add events, conditions or actions as those require a specific implementation via code.',
        'inputs' => [
          'free_text' => [
            'name' => 'Prompt',
            'type' => 'string',
            'description' => 'The prompt to create, edit or ask questions about Event-Condition-Actions models.',
            'default_value' => '',
          ],
        ],
        'outputs' => [
          'answers' => [
            'description' => 'The answers to the questions asked about the Event-Condition-Action model.',
            'type' => 'string',
          ],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function determineSolvability(): int {
    parent::determineSolvability();
    $type = $this->determineTypeOfTask();

    return match ($type) {
      'create', 'edit' => AiAgentInterface::JOB_SOLVABLE,
      'info' => AiAgentInterface::JOB_SHOULD_ANSWER_QUESTION,
      'fail' => AiAgentInterface::JOB_NEEDS_ANSWERS,
      default => AiAgentInterface::JOB_NOT_SOLVABLE,
    };
  }

  /**
   * {@inheritdoc}
   */
  public function solve(): array|string {
    if (isset($this->dto['setup_agent']) && $this->dto['setup_agent'] === TRUE) {
      parent::determineSolvability();
    }

    switch (Arr::get($this->dto, 'data.0.action')) {
      case 'create':
      case 'edit':
        $this->buildModel();
        break;

      case 'info':
        $log = $this->answerQuestion();
        $this->dto['logs'][] = $log;
        break;
    }

    return Arr::join($this->dto['logs'], "\n");
  }

  /**
   * {@inheritdoc}
   */
  public function answerQuestion(): string|TranslatableMarkup {
    if (!$this->currentUser->hasPermission('administer eca')) {
      return $this->t('You do not have permission to do this.');
    }

    $this->dataProvider->setViewMode(DataViewModeEnum::Full);

    $context = [];
    if (!empty($this->model)) {
      $context['The information of the model, in JSON format'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels([$this->model->id()])));
    }
    elseif (!empty($this->data[0]['component_ids'])) {
      $context['The details about the components'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents($this->data[0]['component_ids'])));
    }

    if (empty($context)) {
      return $this->t('Sorry, I could not answer your question without anymore context.');
    }

    // Perform the prompt.
    $data = $this->agentHelper->runSubAgent('answerQuestion', $context);

    if (isset($data[0]['answer'])) {
      $answer = array_map(function ($dataPoint) {
        return $dataPoint['answer'];
      }, $data);

      return implode("\n", $answer);
    }

    return $this->t('Sorry, I got no answers for you.');
  }

  /**
   * {@inheritdoc}
   */
  public function getHelp(): MarkupInterface {
    return $this->t('This agent can figure out event-condition-action models of a file. Just upload and ask.');
  }

  /**
   * {@inheritdoc}
   */
  public function askQuestion(): array {
    return (array) Arr::get($this->dto, 'questions');
  }

  /**
   * {@inheritdoc}
   */
  public function approveSolution(): void {
    $this->dto = Arr::set($this->dto, 'data.0.action', 'create');
  }

  /**
   * {@inheritdoc}
   */
  public function setTask(TaskInterface $task): void {
    parent::setTask($task);

    $this->dto['task_description'] = $task->getDescription();
  }

  /**
   * Get the data transfer object.
   *
   * @return array
   *   Returns the data transfer object.
   */
  public function getDto(): array {
    return $this->dto;
  }

  /**
   * Set the data transfer object.
   *
   * @param array $dto
   *   The data transfer object.
   */
  public function setDto(array $dto): void {
    $this->dto = $dto;

    if (!empty($this->dto['model_id'])) {
      $this->model = $this->ecaRepository->get($this->dto['model_id']);
    }
  }

  /**
   * Determine the type of task.
   *
   * @return string
   *   Returns the type of task.
   *
   * @throws \Exception
   */
  protected function determineTypeOfTask(): string {
    $context = $this->getFullContextOfTask($this->task);
    $userContext = [
      'A summary of the existing models' => sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels())),
      'The list of available events, conditions and actions, in JSON format' => sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents())),
    ];
    if (!empty($this->model)) {
      $context .= "A model already exists, so creation is not possible.";
      $userContext['A summary of the existing models'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getModels([$this->model->id()])));
    }
    $userContext['Task description and if available comments description'] = $context;

    // Prepare and run the prompt by fetching all the relevant info.
    $data = $this->agentHelper->runSubAgent('determineTask', $userContext);

    // Quit early if the returned response isn't what we expected.
    if (empty($data[0]['action'])) {
      $this->dto['questions'][] = 'Sorry, we could not understand what you wanted to do, please try again.';

      return 'fail';
    }

    if (in_array($data[0]['action'], [
      'create',
      'edit',
      'info',
    ])) {
      if (!empty($data[0]['model_id'])) {
        $this->dto['model_id'] = $data[0]['model_id'];
        $this->model = $this->ecaRepository->get($data[0]['model_id']);
      }

      if (!empty($data[0]['feedback'])) {
        $this->dto['feedback'] = $data[0]['feedback'];
      }

      $this->dto['data'] = $data;

      return $data[0]['action'];
    }

    throw new \Exception('Invalid action determined for ECA');
  }

  /**
   * Create a configuration item for ECA.
   */
  protected function buildModel(): void {
    $this->dataProvider->setViewMode(DataViewModeEnum::Full);

    // Prepare the prompt.
    $context = [];

    // The schema of the ECA-config that the LLM should follow.
    $definition = EcaModelDefinition::create();
    $schema = new EcaSchema($definition, $definition->getPropertyDefinitions());
    $schema = $this->serializer->serialize($schema, 'schema_json:json', []);
    $context['JSON Schema of the model'] = sprintf("```json\n%s\n```", $schema);

    // The model that should be edited.
    if (!empty($this->model)) {
      $models = $this->dataProvider->getModels([$this->model->id()]);
      $context['The model to edit'] = sprintf("```json\n%s\n```", Json::encode(reset($models)));
    }

    // Components or plugins that the LLM should use.
    if (Arr::has($this->dto, 'data.0.component_ids')) {
      $componentIds = Arr::get($this->dto, 'data.0.component_ids', []);
      $context['The details about the components'] = sprintf("```json\n%s\n```", Json::encode($this->dataProvider->getComponents($componentIds)));
    }

    // Optional feedback that the previous prompt provided.
    if (Arr::has($this->dto, 'data.0.feedback')) {
      $context['Guidelines'] = Arr::get($this->dto, 'data.0.feedback');
    }

    // Execute it.
    $data = $this->agentHelper->runSubAgent('buildModel', $context);

    $message = Arr::get($data, '0.message', 'Could not create ECA config.');
    if (
      Arr::get($data, '0.action', 'fail') !== 'build'
      || !Arr::has($data, '0.model')
    ) {
      throw new \Exception($message);
    }

    $eca = $this->ecaRepository->build(Json::decode(Arr::get($data, '0.model')), TRUE, $this->model?->id());

    $this->dto['logs'][] = $this->model ? $this->t('Model "@model" was altered.', ['@model' => $this->model->label()]) : $this->t('A new model was created: "@model".', ['@model' => $eca->label()]);
    $this->dto['logs'][] = $message;
  }

}
