<?php

namespace Drupal\ai_eca_agents\Services\EcaRepository;

use Drupal\ai_eca_agents\EntityViolationException;
use Drupal\ai_eca_agents\MissingEventException;
use Drupal\ai_eca_agents\Services\ModelMapper\ModelMapperInterface;
use Drupal\Component\Utility\Random;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\eca\Entity\Eca;

/**
 * Repository for the ECA entity type.
 */
class EcaRepository implements EcaRepositoryInterface {

  /**
   * Constructs an EcaHelper-instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\ai_eca_agents\Services\ModelMapper\ModelMapperInterface $modelMapper
   *   The model mapper.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typedDataManager
   *   The typed data manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModelMapperInterface $modelMapper,
    protected TypedDataManagerInterface $typedDataManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function get(string $id): ?Eca {
    return $this->entityTypeManager->getStorage('eca')
      ->load($id);
  }

  /**
   * {@inheritdoc}
   */
  public function build(array $data, bool $save = TRUE, ?string $id = NULL): Eca {
    /** @var \Drupal\eca\Entity\EcaStorage $storage */
    $storage = $this->entityTypeManager->getStorage('eca');
    /** @var \Drupal\eca\Entity\Eca $eca */
    $eca = $storage->create();
    if (!empty($id)) {
      $eca = $storage->load($id);
    }

    // Convert the given data to the ECA-model.
    $model = $this->modelMapper->fromPayload($data);

    // Map the model to the entity.
    $random = new Random();
    $idFallback = $model->get('model_id')?->getString() ?? sprintf('process_%s', $random->name(7));
    $eca->set('id', $id ?? $idFallback);
    $eca->set('label', $model->get('label')->getString());
    $eca->set('modeller', 'fallback');
    $eca->set('version', $eca->get('version') ?? '0.0.1');
    $eca->setStatus(FALSE);
    $eca->resetComponents();

    // Set events.
    /** @var \Drupal\ai_eca_agents\Plugin\DataType\EcaPlugin $plugin */
    foreach ($model->get('events') as $plugin) {
      $successors = $plugin->get('successors')->getValue() ?? [];
      foreach ($successors as &$successor) {
        $successor['id'] = $successor['element_id'];
        $successor['condition'] ??= '';
        unset($successor['element_id']);
      }

      $eca->addEvent(
        $plugin->get('element_id')->getString(),
        $plugin->get('plugin_id')->getString(),
        $plugin->get('label')->getString(),
        $plugin->get('configuration')->getValue(),
        $successors
      );
    }

    // Set conditions.
    /** @var \Drupal\ai_eca_agents\Plugin\DataType\EcaPlugin $plugin */
    foreach ($model->get('conditions') as $plugin) {
      $eca->addCondition(
        $plugin->get('element_id')->getString(),
        $plugin->get('plugin_id')->getString(),
        $plugin->get('configuration')->getValue(),
      );
    }

    // Set gateways.
    /** @var \Drupal\ai_eca_agents\Plugin\DataType\EcaGateway $plugin */
    foreach ($model->get('gateways') as $plugin) {
      $successors = $plugin->get('successors')->getValue() ?? [];
      foreach ($successors as &$successor) {
        $successor['id'] = $successor['element_id'];
        $successor['condition'] ??= '';
        unset($successor['element_id']);
      }

      $eca->addGateway(
        $plugin->get('gateway_id')->getString(),
        $plugin->get('type')->getValue(),
        $successors
      );
    }

    // Set actions.
    /** @var \Drupal\ai_eca_agents\Plugin\DataType\EcaGateway $plugin */
    foreach ($model->get('actions') as $plugin) {
      $successors = $plugin->get('successors')->getValue() ?? [];
      foreach ($successors as &$successor) {
        $successor['id'] = $successor['element_id'];
        $successor['condition'] ??= '';
        unset($successor['element_id']);
      }

      $eca->addAction(
        $plugin->get('element_id')->getString(),
        $plugin->get('plugin_id')->getString(),
        $plugin->get('label')->getString(),
        $plugin->get('configuration')->getValue() ?? [],
        $successors
      );
    }

    // Validate the entity.
    $definition = $this->typedDataManager->createDataDefinition(sprintf('entity:%s', $eca->getEntityTypeId()));
    $violations = $this->typedDataManager->create($definition, $eca)
      ->validate();
    if ($violations->count()) {
      throw new EntityViolationException('', 0, NULL, $violations);
    }
    if (empty($eca->getUsedEvents())) {
      throw new MissingEventException('No events registered.');
    }

    // Save the entity.
    if ($save) {
      $eca->save();
    }

    return $eca;
  }

}
