<?php

namespace Drupal\ai_evaluations\Controller;

use Drupal\ai_assistant_api\AiAssistantInterface;
use Drupal\ai_evaluations\AIEvaluationInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

class Evaluations extends ControllerBase {

  /**
   * Evaluations per assistant route callback.
   *
   * @param \Drupal\ai_assistant_api\AiAssistantInterface $ai_assistant
   *   The assistant to show evaluations for.
   *
   * @return array
   *   Renderable array of evaluations for an assistant.
   */
  public function assistantEvaluations(AiAssistantInterface $ai_assistant): array {
    $output = [];

    $output['summary'] = [
      '#theme' => 'container',
      '#children' => [
        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#value' => $this->t('Summary'),
        ],
        'upvotes' => [
          '#markup' => new TranslatableMarkup('<p>Total upvotes: @count</p>', ['@count' => $this->countUpvotes($ai_assistant)]),
        ],
        'downvotes' => [
          '#markup' => new TranslatableMarkup('<p>Total downvotes: @count</p>', ['@count' => $this->countDownvotes($ai_assistant)]),
        ],
      ],
    ];

    $output['view'] = [
      '#type' => 'view',
      '#name' => 'evaluations',
      '#arguments' => $ai_assistant ? [$ai_assistant->id()] : NULL,
    ];
    return $output;
  }

  /**
   * Specific Evaluation route callback.
   *
   * @param AIEvaluationInterface $ai_evaluation
   *   The Evaluation being displayed.
   *
   * @return array
   *   Renderable array of evaluations for an assistant.
   */
  public function assistantEvaluation(AIEvaluationInterface $ai_evaluation): array {
    return $this->entityTypeManager()->getViewBuilder($ai_evaluation->getEntityTypeId())->view($ai_evaluation);
  }

  /**
   * Specific Evaluation route callback.
   *
   * @param AIEvaluationInterface $ai_evaluation
   *   The Evaluation being displayed.
   *
   * @return array|\Symfony\Component\HttpFoundation\RedirectResponse
   *   Renderable array of evaluations for an assistant or a redirect response.
   */
  public function evaluation(AIEvaluationInterface $ai_evaluation): array|RedirectResponse {
    if (!$ai_evaluation->get('assistant')->isEmpty()) {
      return $this->redirect('ai_evaluations.ai_assistant.evaluation', [
        'ai_assistant' => $ai_evaluation->get('assistant')->target_id,
        'ai_evaluation' => $ai_evaluation->id(),
      ]);
    }
    return $this->assistantEvaluation($ai_evaluation);
  }
  /**
   * Export route handler for CSV export of Evaluations.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response that triggers a file download with Evaluations data.
   */
  public function export(): Response {
    $evaluation_rows = $this->getEvaluationsRows();
    $file_contents = implode("\n", $evaluation_rows);

    $response = new Response($file_contents);

    $date_string = (new DrupalDateTime())->format('ymd');
    $disposition = $response->headers->makeDisposition(ResponseHeaderBag::DISPOSITION_ATTACHMENT, "evaluations-{$date_string}.csv");
    $response->headers->set('Content-type', 'text/csv');
    $response->headers->set('Content-Disposition', $disposition);
    $response->headers->set('Content-Transfer-Encoding', 'binary');
    $response->headers->set('Content-length', strlen($file_contents));
    return $response;
  }

  /**
   * Get the number of upvotes for an assistant.
   *
   * @param \Drupal\ai_assistant_api\AiAssistantInterface|NULL $ai_assistant
   *   The assistant to get upvotes for.
   *
   * @return int
   *   The number of upvotes.
   */
  protected function countUpvotes(?AiAssistantInterface $ai_assistant): int {
    $query = $this->entityTypeManager()->getStorage('ai_evaluation')->getQuery();
    if ($ai_assistant) {
      $query->condition('assistant', $ai_assistant->id());
    }
    return $query->accessCheck(FALSE)
      ->condition('value', 1)
      ->count()->execute();
  }

  /**
   * Get the number of downvotes for an assistant.
   *
   * @param \Drupal\ai_assistant_api\AiAssistantInterface|NULL $ai_assistant
   *   The assistant to get downvotes for.
   *
   * @return int
   *   The number of downvotes.
   */
  protected function countDownvotes(?AiAssistantInterface $ai_assistant): int {
    $query = $this->entityTypeManager()->getStorage('ai_evaluation')->getQuery();
    if ($ai_assistant) {
      $query->condition('assistant', $ai_assistant->id());
    }
    return $query->accessCheck(FALSE)
      ->condition('value', 0)
      ->count()->execute();
  }

  /**
   * Get the evaluation rows suitable for export.
   *
   * @return array
   *   An array of Evaluations prepared for export.
   */
  protected function getEvaluationsRows(): array {
    $storage = $this->entityTypeManager()->getStorage('ai_evaluation');
    $query = $storage->getQuery();
    $query->accessCheck(FALSE);
    $ids = $query->execute();

    $header = [];
    $rows = [];
    foreach (array_chunk($ids, 10) as $chunk) {
      /** @var \Drupal\ai_evaluations\AIEvaluationInterface $evaluation */
      foreach ($storage->loadMultiple($chunk) as $evaluation) {
        $row = [];
        foreach ($evaluation->getFieldDefinitions() as $field) {
          // Skip the ID (but not UUID) as it may already be taken.
          if ($field->getName() === 'id') {
            continue;
          }
          // Skip entity reference fields as they won't exist when imported.
          if ($field->getType() === 'entity_reference') {
            continue;
          }

          $header[$field->getName()] = $field->getName();
          $row[$field->getName()] = base64_encode(serialize($evaluation->get($field->getName())->getValue()));
        }
        $rows[] = implode(',', $row);
      }
    }

    return [implode(',', $header)] + $rows;
  }

}
