<?php

namespace Drupal\ai_evaluations\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\ReplaceCommand;
use Drupal\Core\Controller\ControllerBase;

/**
 * A Vote controller to receive AI Evaluation votes.
 */
class Vote extends ControllerBase {

  /**
   * Record an AI Evaluation vote from a request.
   *
   * @param string $assistant_id
   *   The ID of the AI Assistant this vote is for.
   * @param string $thread_id
   *   The ID of the relevant thread for this AI Assistant.
   * @param mixed $vote
   *   The value of the vote for the evaluation.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   An AjaxResponse with command for replacing the vote HTML.
   */
  public function vote(string $assistant_id, string $thread_id, mixed $vote): AjaxResponse {
    /** @var \Drupal\ai_assistant_api\AiAssistantApiRunner $ai_runner */
    $ai_runner = \Drupal::service('ai_assistant_api.runner');
    $assistant = $this->entityTypeManager()->getStorage('ai_assistant')->load($assistant_id);
    $ai_runner->setAssistant($assistant);
    $ai_runner->setThreadsKey($thread_id);

    // Store all config as it is now for evaluation purposes. This way, if the
    // assistant is updated we still have a log of the config that the
    // evaluation is for.
    $this->entityTypeManager()->getStorage('ai_evaluation')->create([
      'bundle' => 'vote',
      'uid' => $this->currentUser()->id(),
      'assistant' => $assistant->id(),
      'value' => $vote,
      'messages' => [$ai_runner->getMessageHistory()],
      'system_role' => $assistant->get('system_role'),
      'preprompt_instructions' => $assistant->get('preprompt_instructions'),
      'pre_action_prompt' => $assistant->get('pre_action_prompt'),
      'assistant_message' => $assistant->get('assistant_message'),
      'thread_id' => $thread_id,
      'model_config' => [
        'provider' => $assistant->get('llm_provider'),
        'model' => $assistant->get('llm_model'),
      ] + ['config' => $assistant->get('llm_configuration')],
    ])->save();


    // Create a response that prevents further submissions.
    $response = new AjaxResponse();
    $response->addCommand(new ReplaceCommand('.chat-message-evaluation', $this->getVotedMarkup()));
    return $response;
  }

  /**
   * Get a render array to replace the upvote/downvote HTML with.
   *
   * @return array
   *   The render array with evaluation logged content.
   */
  public function getVotedMarkup(): array {
    return [
      '#theme' => 'ai_evaluations_logged',
      '#message' => $this->t('Your evaluation has been logged.'),
    ];
  }

}
