<?php

declare(strict_types=1);

namespace Drupal\ai_evaluations\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_evaluations\AIAgentResultInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the AI Agent Result entity class.
 *
 * @ContentEntityType(
 *   id = "ai_agent_result",
 *   label = @Translation("AI Agent Result"),
 *   label_collection = @Translation("AI Agent Results"),
 *   label_singular = @Translation("AI Agent Result"),
 *   label_plural = @Translation("AI Agent Results"),
 *   label_count = @PluralTranslation(
 *     singular = "@count AI Agent Results",
 *     plural = "@count AI Agent Results",
 *   ),
 *   handlers = {
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\ai_evaluations\Form\AIAgentResultForm",
 *       "edit" = "Drupal\ai_evaluations\Form\AIAgentResultForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\ai_evaluations\Routing\AIAgentResultHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "ai_agent_result",
 *   admin_permission = "administer ai_agent_result",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *   },
 *   links = {
 *     "collection" = "/admin/content/ai-agent-result",
 *     "add-form" = "/admin/config/ai/ai-agent-results/add",
 *     "canonical" = "/admin/config/ai/ai-agent-results/{ai_agent_result}",
 *     "edit-form" = "/admin/config/ai/ai-agent-results/{ai_agent_result}",
 *     "delete-form" = "/admin/config/ai/ai-agent-results/{ai_agent_result}/delete",
 *     "delete-multiple-form" = "/admin/content/ai-agent-result/delete-multiple",
 *   },
 *   field_ui_base_route = "entity.ai_agent_result.settings",
 * )
 */
final class AIAgentResult extends ContentEntityBase implements AIAgentResultInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['label'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Label'))
      ->setDescription(t('The label of the Agent Decision entity.'))
      ->setRequired(TRUE)
      ->setSettings([
        'default_value' => '',
        'max_length' => 255,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Authored on'))
      ->setDescription(new TranslatableMarkup('The time that the AI Agent Result was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(new TranslatableMarkup('Changed'))
      ->setDescription(new TranslatableMarkup('The time that the AI Agent Result was last edited.'));

    $fields['ai_evaluation'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('AI Evaluation'))
      ->setSetting('target_type', 'ai_evaluation')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['log_status'] = BaseFieldDefinition::create('list_string')
      ->setLabel(t('Log Status'))
      ->setRequired(TRUE)
      ->setDescription(t('The status of the log.'))
      ->setSettings([
        'default_value' => 'notice',
        'allowed_values' => [
          'notice' => 'Notice',
          'warning' => 'Warning',
          'error' => 'Error',
        ],
      ])
      ->setDisplayOptions('form', [
        'type' => 'list_string',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['thread_id'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Thread ID'))
      ->setDescription(t('The thread id.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['ai_provider_info'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('AI Provider Info'))
      ->setDescription(t('The provider and model etc.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['prompt_used'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Prompt Used'))
      ->setDescription(t('The prompt used for the decision.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['question'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Question'))
      ->setDescription(t('The question to the agent.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['response_given'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Response Given'))
      ->setDescription(t('The response given by the agent.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    $fields['detailed_output'] = BaseFieldDefinition::create('string_long')
      ->setLabel(t('Detailed Output'))
      ->setDescription(t('The detailed output of the decision.'))
      ->setSettings([
        'default_value' => '',
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textarea',
        'weight' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', FALSE);

    return $fields;
  }

}
