<?php

declare(strict_types=1);

namespace Drupal\ai_evaluations\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai_evaluations\AIEvaluationInterface;
use Drupal\user\EntityOwnerTrait;

/**
 * Defines the AI evaluation entity class.
 *
 * @ContentEntityType(
 *   id = "ai_evaluation",
 *   label = @Translation("AI Evaluation"),
 *   label_collection = @Translation("AI Evaluations"),
 *   label_singular = @Translation("AI evaluation"),
 *   label_plural = @Translation("AI evaluations"),
 *   label_count = @PluralTranslation(
 *     singular = "@count AI evaluations",
 *     plural = "@count AI evaluations",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\ai_evaluations\AIEvaluationListBuilder",
 *     "views_data" = "Drupal\views\EntityViewsData",
 *     "form" = {
 *       "add" = "Drupal\ai_evaluations\Form\AIEvaluationForm",
 *       "edit" = "Drupal\ai_evaluations\Form\AIEvaluationForm",
 *       "delete" = "Drupal\Core\Entity\ContentEntityDeleteForm",
 *       "delete-multiple-confirm" = "Drupal\Core\Entity\Form\DeleteMultipleForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\ai_evaluations\Routing\AIEvaluationHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "ai_evaluation",
 *   admin_permission = "administer ai_evaluation",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "id",
 *     "uuid" = "uuid",
 *     "owner" = "uid",
 *     "bundle" = "bundle",
 *   },
 *   links = {
 *     "collection" = "/admin/config/ai/ai-evaluation/evaluations",
 *     "add-form" = "/admin/config/ai/ai-evaluations/add",
 *     "canonical" = "/admin/config/ai/ai-evaluation/{ai_evaluation}",
 *     "edit-form" = "/admin/config/ai/ai-evaluation/{ai_evaluation}/edit",
 *     "delete-form" = "/admin/config/ai/ai-evaluations/{ai_evaluation}/delete",
 *     "delete-multiple-form" = "/admin/content/ai-evaluation/delete-multiple",
 *   },
 *   bundle_entity_type = "ai_evaluation_type",
 *   field_ui_base_route = "ai_evaluations.ai_evaluation_type.edit_form",
 * )
 */
final class AIEvaluation extends ContentEntityBase implements AIEvaluationInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the anonymous user the owner.
      $this->setOwnerId(0);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Author'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => 60,
          'placeholder' => '',
        ],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(new TranslatableMarkup('Authored on'))
      ->setDescription(new TranslatableMarkup('The time that the AI evaluation was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(new TranslatableMarkup('Changed'))
      ->setDescription(new TranslatableMarkup('The time that the AI evaluation was last edited.'));

    $fields['assistant'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(new TranslatableMarkup('Assistant'))
      ->setSetting('target_type', 'ai_assistant')
      ->setDisplayConfigurable('view', TRUE);

    $fields['messages'] = BaseFieldDefinition::create('map')
      ->setLabel(new TranslatableMarkup('Messages'))
      ->setDisplayConfigurable('view', TRUE);

    $fields['system_role'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('System role'))
      ->setCardinality(1)
      ->setDisplayConfigurable('view', TRUE);

    $fields['preprompt_instructions'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Pre-prompt instructions'))
      ->setCardinality(1)
      ->setDisplayConfigurable('view', TRUE);

    $fields['pre_action_prompt'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Pre-action prompt'))
      ->setCardinality(1)
      ->setDisplayConfigurable('view', TRUE);

    $fields['assistant_message'] = BaseFieldDefinition::create('string_long')
      ->setLabel(new TranslatableMarkup('Assistant message'))
      ->setCardinality(1)
      ->setDisplayConfigurable('view', TRUE);

    $fields['model_config'] = BaseFieldDefinition::create('map')
      ->setLabel(new TranslatableMarkup('Model config'))
      ->setDisplayConfigurable('view', TRUE);

    $fields['thread_id'] = BaseFieldDefinition::create('string')
      ->setLabel(new TranslatableMarkup('Thread ID'))
      ->setDisplayConfigurable('view', TRUE);

    $fields['value'] = BaseFieldDefinition::create('integer')
      ->setLabel(new TranslatableMarkup('Value'))
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE) {
    parent::postSave($storage, $update);

    // Connect the agent records to the AI evaluation.
    if ($this->thread_id->value) {
      $agent_storage = $this->entityTypeManager()->getStorage('ai_agent_result');
      $query = $agent_storage->getQuery();
      $query->condition('thread_id', $this->thread_id->value);
      $query->condition('ai_evaluation', NULL, 'IS NULL');
      $query->accessCheck(FALSE);
      $result = $query->execute();
      foreach ($result as $id) {
        /** @var \Drupal\ai_evaluations\Entity\AIAgentResult */
        $agent = $agent_storage->load($id);
        $agent->set('ai_evaluation', $this->id());
        $agent->save();
      }
    }
  }

}
