<?php

namespace Drupal\ai_evaluations\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Formatter that renders Map fields for AI Evaluation.
 */
#[FieldFormatter(
  id: 'ai_map',
  label: new TranslatableMarkup('AI Map'),
  field_types: [
    'map',
  ],
)]
class AiMap extends FormatterBase {

  const AI_MAP_TYPE_MODEL_CONFIG = 'model_config';
  const AI_MAP_TYPE_MESSAGES_ALL = 'messages';

  const AI_MAP_TYPE_MESSAGES_FIRST_USER = 'first_user';
  const AI_MAP_TYPE_MESSAGES_LAST_USER = 'last_user';
  const AI_MAP_TYPE_MESSAGES_LAST_ASSISTANT = 'last_assistant';

  public static function defaultSettings() {
    $settings = parent::defaultSettings();

    $settings['map_type'] = NULL;
    $settings['messages_type'] = self::AI_MAP_TYPE_MESSAGES_ALL;

    return $settings;
  }

  public function settingsForm(array $form, FormStateInterface $form_state) {
    $form = parent::settingsForm($form, $form_state);

    $form['map_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Map type'),
      '#default_value' => $this->getSetting('map_type'),
      '#options' => [
        self::AI_MAP_TYPE_MESSAGES_ALL => $this->t('All messages'),
        self::AI_MAP_TYPE_MODEL_CONFIG => $this->t('Model config'),
        self::AI_MAP_TYPE_MESSAGES_FIRST_USER => $this->t('First user message'),
        self::AI_MAP_TYPE_MESSAGES_LAST_USER => $this->t('Last user message'),
        self::AI_MAP_TYPE_MESSAGES_LAST_ASSISTANT => $this->t('Last assistant message'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      $value = $item->getValue();

      switch ($this->getSetting('map_type')) {
        case self::AI_MAP_TYPE_MESSAGES_ALL:
          $role = $value['role'];
          $message = $value['message'];

          $elements[$delta] = [
            '#markup' => <<<EOT
<div>
<span>Role: $role</span>
<span>Message: $message</span>
</div>
EOT
          ];
          break;

        case self::AI_MAP_TYPE_MESSAGES_FIRST_USER:
          if (empty($elements) && $value['role'] === 'user') {
            $elements[$delta] = [
              '#markup' => $value['message'] ?? '',
            ];
          }
          break;

        case self::AI_MAP_TYPE_MESSAGES_LAST_USER:
          if ($value['role'] === 'user') {
            $elements = [[
              '#markup' => $value['message'] ?? '',
            ]];
          }
          break;

        case self::AI_MAP_TYPE_MESSAGES_LAST_ASSISTANT:
          if ($value['role'] === 'assistant') {
            $elements = [[
              '#markup' => $value['message'] ?? '',
            ]];
          }
          break;

        case self::AI_MAP_TYPE_MODEL_CONFIG:
          $provider = $value['provider'] ?? '';
          $model = $value['model'] ?? '';
          $config = '';
          foreach ($value['config'] as $config_key => $config_value) {
            $config .= '<div>' . str_replace('_', ' ', ucfirst($config_key)) . ': ' . $config_value . '</div>';
          }

          $elements[$delta] = [
            '#markup' => <<<EOT
<div>
<span>Provider: $provider</span>
<span>Model: $model</span>
<span>Config: $config</span>
</div>
EOT
          ];
          break;
      }
    }

    return $elements;
  }

}
