<?php

namespace Drupal\Tests\ai_integration_eca_agents\Kernel;

use Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Spatie\Snapshots\MatchesSnapshots;
use Symfony\Component\Serializer\Normalizer\NormalizerInterface;

/**
 * Tests various input data for generation ECA Model typed data.
 *
 * @group ai_integration_eca_agents
 */
class ModelMapperTest extends AiEcaAgentsKernelTestBase {

  use MatchesSnapshots;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai_integration_eca',
    'ai_integration_eca_agents',
    'eca',
    'eca_base',
    'eca_content',
    'eca_user',
    'eca_test_model_cross_ref',
    'eca_test_model_entity_basics',
    'eca_test_model_set_field_value',
    'field',
    'node',
    'text',
    'token',
    'user',
    'serialization',
    'schemata',
    'schemata_json_schema',
    'system',
    'taxonomy',
  ];

  /**
   * The model mapper.
   *
   * @var \Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface|null
   */
  protected ?ModelMapperInterface $modelMapper;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|null
   */
  protected ?EntityTypeManagerInterface $entityTypeManager;

  /**
   * The normalizer.
   *
   * @var \Symfony\Component\Serializer\Normalizer\NormalizerInterface|null
   */
  protected ?NormalizerInterface $normalizer;

  /**
   * Build an ECA Model type data object with the provided payloads.
   *
   * @dataProvider payloadProvider
   */
  public function testMappingFromPayload(array $payload, array $assertions, ?string $id = NULL, ?string $errorMessage = NULL): void {
    if (!empty($errorMessage)) {
      $this->expectExceptionMessage($errorMessage);
    }

    $model = $this->modelMapper->fromPayload($payload);

    foreach ($assertions as $property => $expected) {
      switch (TRUE) {
        case is_int($expected):
          $this->assertCount($expected, $model->get($property)->getValue());
          break;

        case is_string($expected):
          $this->assertEquals($expected, $model->get($property)->getString());
          break;
      }
    }
  }

  /**
   * Build an ECA Model data type object with the provided entities.
   *
   * @dataProvider entityProvider
   */
  public function testMappingFromEntity(string $entityId): void {
    /** @var \Drupal\eca\Entity\Eca $entity */
    $entity = $this->entityTypeManager->getStorage('eca')
      ->load($entityId);
    $model = $this->modelMapper->fromEntity($entity);
    $data = $this->normalizer->normalize($model);

    $this->assertMatchesJsonSnapshot(json_encode($data, JSON_PRETTY_PRINT));
  }

  /**
   * Generate different sets of ECA entity IDs.
   *
   * @return \Generator
   *   Returns a collection of ECA entity IDs.
   */
  public static function entityProvider(): \Generator {
    yield [
      'eca_test_0001',
    ];

    yield [
      'eca_test_0002',
    ];

    yield [
      'eca_test_0009',
    ];
  }

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->modelMapper = \Drupal::service('ai_integration_eca_agents.services.model_mapper');
    $this->entityTypeManager = \Drupal::entityTypeManager();
    $this->normalizer = \Drupal::service('serializer');
  }

}
