<?php

namespace Drupal\ai_integration_eca_agents\Services\EcaRepository;

use Drupal\ai_integration_eca_agents\EntityViolationException;
use Drupal\ai_integration_eca_agents\MissingEventException;
use Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface;
use Drupal\Component\Utility\Random;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\TypedData\TypedDataManagerInterface;
use Drupal\eca\Entity\Eca;

/**
 * Repository for the ECA entity type.
 */
class EcaRepository implements EcaRepositoryInterface {

  /**
   * Indicates if the legacy ECA module v2 is installed.
   *
   * @var bool
   */
  private bool $legacyEca = FALSE;

  /**
   * Constructs an EcaHelper-instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface $modelMapper
   *   The model mapper.
   * @param \Drupal\Core\TypedData\TypedDataManagerInterface $typedDataManager
   *   The typed data manager.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ModelMapperInterface $modelMapper,
    protected TypedDataManagerInterface $typedDataManager,
  ) {
    $this->legacyEca = !class_exists('Drupal\eca_ui\Plugin\ModelerApiModelOwner\Eca');
  }

  /**
   * {@inheritdoc}
   */
  public function get(string $id): ?Eca {
    $eca = $this->entityTypeManager->getStorage('eca')->load($id);
    return $eca instanceof Eca ? $eca : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function build(array $data, bool $save = TRUE, ?string $id = NULL): Eca {
    /** @var \Drupal\eca\Entity\EcaStorage $storage */
    $storage = $this->entityTypeManager->getStorage('eca');
    /** @var \Drupal\eca\Entity\Eca $eca */
    $eca = $storage->create();
    if (!empty($id)) {
      /** @var \Drupal\eca\Entity\Eca $eca */
      $eca = $storage->load($id);
    }

    // Convert the given data to the ECA-model.
    $model = $this->modelMapper->fromPayload($data);

    // Map the model to the entity.
    $random = new Random();
    $idFallback = $model->get('model_id')->getString();
    if (empty($idFallback)) {
      $idFallback = sprintf('process_%s', $random->name(7));
    }
    $eca->set('id', $id ?? $idFallback);
    $version = $model->get('version')->getString();
    if ($version === '') {
      $version = '0.0.1';
    }
    if ($this->legacyEca) {
      $eca->set('label', $model->get('label')->getString());
      $eca->set('version', $version);
    }
    else {
      $eca->setThirdPartySetting('modeler_api', 'label', $model->get('label')->getString());
      $eca->setThirdPartySetting('modeler_api', 'version', $version);
    }
    $eca->setStatus(FALSE);
    $eca->resetComponents();

    // Set events.
    $events = $model->get('events');
    if (is_iterable($events)) {
      /** @var \Drupal\ai_integration_eca_agents\Plugin\DataType\EcaPlugin $plugin */
      foreach ($events as $plugin) {
        $successors = $plugin->get('successors')->getValue() ?? [];
        foreach ($successors as &$successor) {
          $successor['id'] = $successor['element_id'];
          $successor['condition'] ??= '';
          unset($successor['element_id']);
        }

        $eca->addEvent(
          $plugin->get('element_id')->getString(),
          $plugin->get('plugin_id')->getString(),
          $plugin->get('label')->getString(),
          $plugin->get('configuration')->getValue() ?? [],
          $successors
        );
      }
    }

    // Set conditions.
    $conditions = $model->get('conditions');
    if (is_iterable($conditions)) {
      /** @var \Drupal\ai_integration_eca_agents\Plugin\DataType\EcaPlugin $plugin */
      foreach ($conditions as $plugin) {
        if ($this->legacyEca) {
          $eca->addCondition(
            $plugin->get('element_id')->getString(),
            $plugin->get('plugin_id')->getString(),
            $plugin->get('configuration')->getValue() ?? [],
          );
        }
        else {
          $eca->addCondition(
            $plugin->get('element_id')->getString(),
            $plugin->get('plugin_id')->getString(),
            // @todo $plugin->get('label')->getString().
            '',
            $plugin->get('configuration')->getValue() ?? [],
          );
        }
      }
    }

    // Set gateways.
    $gateways = $model->get('gateways');
    if (is_iterable($gateways)) {
      /** @var \Drupal\ai_integration_eca_agents\Plugin\DataType\EcaGateway $plugin */
      foreach ($gateways as $plugin) {
        $successors = $plugin->get('successors')->getValue() ?? [];
        foreach ($successors as &$successor) {
          $successor['id'] = $successor['element_id'];
          $successor['condition'] ??= '';
          unset($successor['element_id']);
        }

        $eca->addGateway(
          $plugin->get('gateway_id')->getString(),
          $plugin->get('type')->getValue(),
          $successors
        );
      }
    }

    // Set actions.
    $actions = $model->get('actions');
    if (is_iterable($actions)) {
      /** @var \Drupal\ai_integration_eca_agents\Plugin\DataType\EcaGateway $plugin */
      foreach ($actions as $plugin) {
        $successors = $plugin->get('successors')->getValue() ?? [];
        foreach ($successors as &$successor) {
          $successor['id'] = $successor['element_id'];
          $successor['condition'] ??= '';
          unset($successor['element_id']);
        }

        $eca->addAction(
          $plugin->get('element_id')->getString(),
          $plugin->get('plugin_id')->getString(),
          $plugin->get('label')->getString(),
          $plugin->get('configuration')->getValue() ?? [],
          $successors
        );
      }
    }

    // Validate the entity.
    $definition = $this->typedDataManager->createDataDefinition(sprintf('entity:%s', $eca->getEntityTypeId()));
    $violations = $this->typedDataManager->create($definition, $eca)->validate();
    if ($violations->count()) {
      throw new EntityViolationException('', 0, NULL, $violations);
    }
    if (empty($eca->getUsedEvents())) {
      throw new MissingEventException('No events registered.');
    }

    // Save the entity.
    if ($save) {
      $eca->save();
    }

    return $eca;
  }

}
