<?php

namespace Drupal\ai_integration_eca_agents\Plugin\AiAgentValidation;

use Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface;
use Drupal\Component\Serialization\Json;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\Service\PromptJsonDecoder\PromptJsonDecoderInterface;
use Drupal\ai_agents\Attribute\AiAgentValidation;
use Drupal\ai_agents\Exception\AgentRetryableValidationException;
use Drupal\ai_agents\PluginBase\AiAgentValidationPluginBase;
use Drupal\Core\TypedData\Exception\MissingDataException;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The validation of the ECA agent.
 */
#[AiAgentValidation(
  id: 'eca_validation',
  label: new TranslatableMarkup('ECA Validation'),
)]
final class EcaValidation extends AiAgentValidationPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The JSON-decoder of the prompt.
   *
   * @var \Drupal\ai\Service\PromptJsonDecoder\PromptJsonDecoderInterface
   */
  protected PromptJsonDecoderInterface $promptJsonDecoder;

  /**
   * The model mapper.
   *
   * @var \Drupal\ai_integration_eca_agents\Services\ModelMapper\ModelMapperInterface
   */
  protected ModelMapperInterface $modelMapper;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): EcaValidation {
    $instance = new EcaValidation($configuration, $plugin_id, $plugin_definition);
    $instance->promptJsonDecoder = $container->get('ai.prompt_json_decode');
    $instance->modelMapper = $container->get('ai_integration_eca_agents.services.model_mapper');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function defaultValidation(mixed $data): bool {
    $data = $this->decodeData($data);
    if (empty($data)) {
      throw new AgentRetryableValidationException(
        'The LLM response failed validation: could not decode from JSON.',
        0,
        NULL,
        'You MUST only provide a RFC8259 compliant JSON response.'
      );
    }

    if (!isset($data[0]['message'])) {
      throw new AgentRetryableValidationException(
        'The LLM response failed validation: feedback was not provided.',
        0,
        NULL,
        'You MUST provide a summary of your action or give feedback why you failed.'
      );
    }

    // Stop validation if the LLM failed to return a response.
    if ($data[0]['action'] ?? 'fail') {
      return TRUE;
    }

    // Validate the response against ECA-model schema.
    try {
      $this->modelMapper->fromPayload(Json::decode($data[0]['model'] ?? NULL));
    }
    catch (MissingDataException $e) {
      throw new AgentRetryableValidationException(
        'The LLM response failed validation: the model definition was violated.',
        0,
        NULL,
        sprintf('Fix the following violations: %s.', $e->getMessage())
      );
    }

    return TRUE;
  }

  /**
   * Decodes the source data.
   *
   * @param mixed $source
   *   The source.
   *
   * @return array|null
   *   Returns the decoded data or NULL.
   */
  protected function decodeData(mixed $source): ?array {
    $text = NULL;

    switch (TRUE) {
      case $source instanceof ChatMessage:
        $result = $this->promptJsonDecoder->decode($source);
        return is_array($result) ? $result : $result->toArray();

      case is_string($source):
        $text = $source;
        break;
    }

    return Json::decode($text ?? '');
  }

}
