<?php

namespace Drupal\ai_interpolator\Plugin\QueueWorker;

use Drupal\ai_interpolator\AiInterpolatorRuleRunner;
use Drupal\ai_interpolator\Exceptions\AiInterpolatorRequestErrorException;
use Drupal\ai_interpolator\Exceptions\AiInterpolatorResponseErrorException;
use Drupal\ai_interpolator\Exceptions\AiInterpolatorRuleNotFoundException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelFactory;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Queue\QueueWorkerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A queue worker that fills fields for entities from AI data.
 *
 * @QueueWorker(
 *   id = "ai_interpolator_field_modifier",
 *   title = @Translation("Queue Job to fill in AI produced data"),
 *   cron = {"time" = 1}
 * )
 */
class InterpolatorFieldData extends QueueWorkerBase implements ContainerFactoryPluginInterface {

  /**
   * Runner.
   */
  protected AiInterpolatorRuleRunner $aiRunner;

  /**
   * The Drupal entity manager.
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Drupal logger factory.
   */
  protected LoggerChannelFactory $loggerFactory;

  /**
   * Constructor.
   *
   * @param \Drupal\ai_interpolator\AiInterpolatorRuleRunner $aiRunner
   *   The AI runner.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelFactory $loggerFactory
   *   The logger factory.
   */
  final public function __construct(AiInterpolatorRuleRunner $aiRunner, EntityTypeManagerInterface $entityTypeManager, LoggerChannelFactory $loggerFactory) {
    $this->aiRunner = $aiRunner;
    $this->entityTypeManager = $entityTypeManager;
    $this->loggerFactory = $loggerFactory;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('ai_interpolator.rule_runner'),
      $container->get('entity_type.manager'),
      $container->get('logger.factory')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function processItem($data) {
    $this->loggerFactory->get('ai_interpolator')->info("Cronjob starting to fill out field %field on entity %entity_type with id %id", [
      '%field' => $data['interpolatorConfig']['field_name'],
      '%entity_type' => $data['entity']->getEntityTypeId(),
      '%id' => $data['entity']->id(),
    ]);

    try {
      $success = $this->aiRunner->generateResponse($data['entity'], $data['fieldDefinition'], $data['interpolatorConfig']);
      $this->loggerFactory->get('ai_interpolator')->info("Cronjob finished to fill out field %field on entity %entity_type with id %id", [
        '%field' => $data['interpolatorConfig']['field_name'],
        '%entity_type' => $data['entity']->getEntityTypeId(),
        '%id' => $data['entity']->id(),
      ]);
      return $success;
    }
    catch (AiInterpolatorRuleNotFoundException $e) {
      $this->loggerFactory->get('ai_interpolator')->warning('A rule was not found, message %message', [
        '%message' => $e->getMessage(),
      ]);
    }
    catch (AiInterpolatorRequestErrorException $e) {
      $this->loggerFactory->get('ai_interpolator')->warning('A request error happened, message %message', [
        '%message' => $e->getMessage(),
      ]);
    }
    catch (AiInterpolatorResponseErrorException $e) {
      $this->loggerFactory->get('ai_interpolator')->warning('A response was not correct, message %message', [
        '%message' => $e->getMessage(),
      ]);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('ai_interpolator')->warning('A general error happened why trying to interpolate, message %message', [
        '%message' => $e->getMessage(),
      ]);
    }

  }

}
