<?php

namespace Drupal\Tests\ai_interpolator\Unit;

use Drupal\ai_interpolator\AiFieldRules;
use Drupal\ai_interpolator\PluginManager\AiInterpolatorFieldRuleManager;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\ai_interpolator\AiFieldRules
 * @group ai_interpolator
 */
class AiFieldRulesTest extends UnitTestCase {

  /**
   * The Ai Field Rules under test.
   */
  protected AiFieldRules $fieldRules;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    $fieldRuleManager = $this->getMockBuilder(AiInterpolatorFieldRuleManager::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldRuleManager
      ->method('getDefinitions')
      ->willReturn([
        [
          'id' => 'definition_1',
          'field_rule' => 'boolean',
          'target' => NULL,
        ],
        [
          'id' => 'definition_2',
          'field_rule' => 'boolean',
          'target' => NULL,
        ],
        [
          'id' => 'definition_3_with_target',
          'field_rule' => 'entity_reference',
          'target' => 'taxonomy_term',
        ],
      ]);
    $fieldRuleManager
      ->method('createInstance')
      ->will($this->returnCallback(
        function ($argument) {
          switch ($argument) {
            case 'definition_1':
              return [
                'id' => 'definition_1',
                'field_rule' => 'boolean',
                'target' => NULL,
              ];

            case 'definition_2':
              return [
                'id' => 'definition_2',
                'field_rule' => 'boolean',
                'target' => NULL,
              ];

            case 'definition_3_with_target':
              return [
                'id' => 'definition_3_with_target',
                'field_rule' => 'entity_reference',
                'target' => 'taxonomy_term',
              ];
          }
          return NULL;
        }
      )
    );

    $this->fieldRules = new AiFieldrules($fieldRuleManager);
  }

  /**
   * Test that the find rules returns mockup object.
   */
  public function testFindingRule(): void {
    $this->assertNotNull($this->fieldRules->findRule('definition_1'));
  }

  /**
   * Test that the find rules returns nothing.
   */
  public function testNotFindingRule(): void {
    $this->assertNull($this->fieldRules->findRule('not_found'));
  }

  /**
   * Test that the find rules returns mockup object.
   */
  public function testFindingCandidates(): void {
    $settingsInterface = $this->getMockBuilder(FieldStorageDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $settingsInterface
      ->method('getSettings')
      ->willReturn([
        'target_type' => '',
      ]);
    $fieldInterface = $this->getMockBuilder(FieldDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldInterface
      ->method('getFieldStorageDefinition')
      ->willReturn($settingsInterface);
    $fieldInterface
      ->method('getType')
      ->willReturn('boolean');
    $this->assertArrayHasKey('definition_1', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayHasKey('definition_2', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayNotHasKey('definition_3_with_target', $this->fieldRules->findRuleCandidates($fieldInterface));
  }

  /**
   * Test that the find rules returns nothing.
   */
  public function testNotFindingCandidates(): void {
    $settingsInterface = $this->getMockBuilder(FieldStorageDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $settingsInterface
      ->method('getSettings')
      ->willReturn([
        'target_type' => '',
      ]);
    $fieldInterface = $this->getMockBuilder(FieldDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldInterface
      ->method('getFieldStorageDefinition')
      ->willReturn($settingsInterface);
    $fieldInterface
      ->method('getType')
      ->willReturn('string');
    $this->assertArrayNotHasKey('definition_1', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayNotHasKey('definition_2', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayNotHasKey('definition_3_with_target', $this->fieldRules->findRuleCandidates($fieldInterface));
  }

  /**
   * Test that the find rules returns mockup object for targets.
   */
  public function testFindingTargetCandidates(): void {
    $settingsInterface = $this->getMockBuilder(FieldStorageDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $settingsInterface
      ->method('getSettings')
      ->willReturn([
        'target_type' => 'taxonomy_term',
      ]);
    $fieldInterface = $this->getMockBuilder(FieldDefinitionInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldInterface
      ->method('getFieldStorageDefinition')
      ->willReturn($settingsInterface);
    $fieldInterface
      ->method('getType')
      ->willReturn('entity_reference');
    $this->assertArrayNotHasKey('definition_1', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayNotHasKey('definition_2', $this->fieldRules->findRuleCandidates($fieldInterface));
    $this->assertArrayHasKey('definition_3_with_target', $this->fieldRules->findRuleCandidates($fieldInterface));
  }

}
