<?php

namespace Drupal\Tests\ai_interpolator\Unit;

use Drupal\ai_interpolator\AiInterpolatorEntityModifier;
use Drupal\ai_interpolator\FieldModifiers\BatchProcessing;
use Drupal\ai_interpolator\FieldModifiers\DirectSave;
use Drupal\ai_interpolator\FieldModifiers\QueueWorker;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Field\FieldConfigInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Tests\UnitTestCase;

/**
 * @coversDefaultClass \Drupal\ai_interpolator\AiInterpolatorEntityModifier
 * @group ai_interpolator
 */
class AiInterpolatorEntityModifierTest extends UnitTestCase {

  /**
   * The Entity Modifier under test.
   */
  protected AiInterpolatorEntityModifier $entityModifier;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    // Drupal dependency hell :(.
    $entityFieldManager = $this->getMockBuilder(EntityFieldManagerInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $entityFieldManager
      ->method('getFieldDefinitions')
      ->will($this->returnCallback(
        function ($entityType, $bundle) {
          if ($entityType == 'entity_with_configs' && $bundle == 'bundle_with_configs_not_enabled') {
            $fieldConfigDisabled = $this->getMockBuilder(FieldConfigInterface::class)
              ->disableOriginalConstructor()
              ->getMock();
            $fieldConfigDisabled
              ->method('getThirdPartySetting')
              ->willReturn(FALSE);
            $fieldDefinitionDisabled = $this->getMockBuilder(FieldDefinitionInterface::class)
              ->disableOriginalConstructor()
              ->getMock();
            $fieldDefinitionDisabled
              ->method('getConfig')
              ->willReturn($fieldConfigDisabled);
            return [$fieldDefinitionDisabled];
          }
          elseif ($entityType == 'entity_with_configs' && $bundle == 'bundle_with_configs') {
            $fieldConfigEnabled = $this->getMockBuilder(FieldConfigInterface::class)
              ->disableOriginalConstructor()
              ->getMock();
            $fieldConfigEnabled
              ->method('getThirdPartySetting')
              ->willReturn(TRUE);
            $fieldConfigEnabled
              ->method('getName')
              ->willReturn('groot');
            $fieldConfigEnabled
              ->method('getThirdPartySettings')
              ->willReturn([
                'test_not_catched' => 'hello',
                'interpolator_i_am_groot' => 'I am Groot!',
                'interpolator_fancy' => 'name',
                'interpolator_base_field' => 'description',
                'interpolator_edit_mode' => TRUE,
                'interpolator_worker_type' => 'queue',
              ]);
            $fieldDefinitionEnabled = $this->getMockBuilder(FieldDefinitionInterface::class)
              ->disableOriginalConstructor()
              ->getMock();
            $fieldDefinitionEnabled
              ->method('getConfig')
              ->willReturn($fieldConfigEnabled);
            return [$fieldDefinitionEnabled];
          }
          return [];
        }
      )
    );
    $queueWorker = $this->getMockBuilder(QueueWorker::class)
      ->disableOriginalConstructor()
      ->getMock();
    $directSave = $this->getMockBuilder(DirectSave::class)
      ->disableOriginalConstructor()
      ->getMock();
    $batchSave = $this->getMockBuilder(BatchProcessing::class)
      ->disableOriginalConstructor()
      ->getMock();

    $this->entityModifier = new AiInterpolatorEntityModifier($entityFieldManager, $queueWorker, $directSave, $batchSave);
  }

  /**
   * Test that config entities just fails.
   */
  public function testConfigEntity(): void {
    $configEntity = $this->getMockBuilder(ConfigEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $this->assertFalse($this->entityModifier->saveEntity($configEntity));
  }

  /**
   * Test that loads entity without configs.
   */
  public function testEntityWithoutConfigs(): void {
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_without_configs');

    $this->assertEmpty($this->entityModifier->entityHasConfig($contentEntity));
  }

  /**
   * Test that loads entity with config and bundle without configs.
   */
  public function testEntityWithConfigsWithoutBundle(): void {
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_without_configs');

    $this->assertEmpty($this->entityModifier->entityHasConfig($contentEntity));
  }

  /**
   * Test that loads entity without enabled settings.
   */
  public function testEntityWithConfigsWithBundleDisabled(): void {
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_with_configs_not_enabled');

    $this->assertEmpty($this->entityModifier->entityHasConfig($contentEntity));
  }

  /**
   * Test that loads entity all working.
   */
  public function testEntityWithConfigsWithBundleEnabled(): void {
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_with_configs');

    $returnArray = $this->entityModifier->entityHasConfig($contentEntity);
    $this->assertArrayHasKey('groot', $returnArray);
    $this->assertArrayHasKey('interpolatorConfig', $returnArray['groot']);
    $this->assertArrayHasKey('fieldDefinition', $returnArray['groot']);
    $this->assertInstanceOf(FieldDefinitionInterface::class, $returnArray['groot']['fieldDefinition']);
    $this->assertArrayHasKey('i_am_groot', $returnArray['groot']['interpolatorConfig']);
    $this->assertArrayHasKey('fancy', $returnArray['groot']['interpolatorConfig']);
    $this->assertArrayNotHasKey('test_not_catched', $returnArray['groot']['interpolatorConfig']);
  }

  /**
   * Test that loads entity not working.
   */
  public function testSavingEntityWithoutConfig(): void {
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_with_configs_not_enabled');

    $this->assertFalse($this->entityModifier->saveEntity($contentEntity));
  }

  /**
   * Test that loads entity but empty base field.
   */
  public function testSavingEntityWithConfigWithoutBaseFieldValue(): void {
    $fieldItem = $this->getMockBuilder(FieldItemListInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldItem
      ->method('getValue')
      ->willReturn([]);
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_with_configs');
    $contentEntity
      ->method('get')
      ->willReturn($fieldItem);
    $contentEntity->description = new \stdClass();
    $contentEntity->description->value = '';
    $contentEntity->groot = new \stdClass();
    $contentEntity->groot->value = 'test';

    // It will still be true.
    $this->assertTrue($this->entityModifier->saveEntity($contentEntity));
  }

  /**
   * Test that loads entity but empty base field.
   */
  public function testSavingEntityWithConfigWithBaseFieldValue(): void {
    $fieldItem = $this->getMockBuilder(FieldItemListInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $fieldItem
      ->method('getValue')
      ->willReturn([]);
    $contentEntity = $this->getMockBuilder(ContentEntityInterface::class)
      ->disableOriginalConstructor()
      ->getMock();
    $contentEntity
      ->method('getEntityTypeId')
      ->willReturn('entity_with_configs');
    $contentEntity
      ->method('bundle')
      ->willReturn('bundle_with_configs');
    $contentEntity
      ->method('get')
      ->willReturn($fieldItem);
    $contentEntity->description = new \stdClass();
    $contentEntity->description->value = 'test';
    $contentEntity->groot = new \stdClass();
    $contentEntity->groot->value = 'test';

    // It will still be true.
    $this->assertTrue($this->entityModifier->saveEntity($contentEntity));
  }

}
