<?php

namespace Drupal\ai_interpolator_agent;

use Drupal\ai_interpolator_agent\Entity\AiWorkerAgent;
use Drupal\Core\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\minikanban\Entity\Task;
use Drupal\minikanban_agent\AgentHelper;
use Drupal\minikanban_agent\AgentSolutions\DirectSolution;
use Drupal\minikanban_agent\Exceptions\AgentRunnerException;
use Drupal\minikanban_agent\PluginInterfaces\MinikanbanLlmInterface;

class AgentsHelper {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The agent helper.
   *
   * @var \Drupal\minikanban_agent\AgentHelper
   */
  protected $agentHelper;

  /**
   * The task.
   *
   * @var \Drupal\minikanban\Entity\Task
   */
  protected $task;

  /**
   * The Llm.
   *
   * @var \Drupal\minikanban_agent\PluginInterfaces\MinikanbanLlmInterface
   */
  protected $llm;

  /**
   * AgentsHelper constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    AgentHelper $agentHelper,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->agentHelper = $agentHelper;
  }

  /**
   * Gets all agents.
   *
   * @param string $excludedAgent
   *   An agent to exlude.
   * @param bool $workers
   *   Include workers.
   * @param bool $managers
   *   Include managers.
   * @param string $sort
   *   The sort order.
   * @param string $sortDirection
   *   The sort direction.
   *
   * @return array
   *   An array of all agents.
   */
  public function getAllAgents($excludedAgent = '', $workers = TRUE, $managers = TRUE, $sort = 'label', $sortDirection = 'ASC') {
    $agents = [];
    if ($managers) {
      $storage = $this->entityTypeManager->getStorage('ai_manager_agent');
      $query = $storage->getQuery();
      $query->condition('status', 1);
      $query->sort($sort, $sortDirection);
      if ($excludedAgent) {
        $query->condition('id', $excludedAgent, '<>');
      }
      $ids = $query->execute();
      if ($ids) {
        $agents = $storage->loadMultiple($ids);
      }
    }
    if ($workers) {
      $storage = $this->entityTypeManager->getStorage('ai_worker_agent');
      $query = $storage->getQuery();
      $query->condition('status', 1);
      $query->sort($sort, $sortDirection);
      if ($excludedAgent) {
        $query->condition('id', $excludedAgent, '<>');
      }
      $ids = $query->execute();
      if ($ids) {
        $agents = array_merge($storage->loadMultiple($ids), $agents);
      }
    }
    return $agents;
  }

  /**
   * Get one agent.
   *
   * @param string $id
   *   The agent id.
   *
   * @return \Drupal\Core\Entity\ConfigEntityInterface|null
   *   The agent entity.
   */
  public function getAgent($id) {
    $storage = $this->entityTypeManager->getStorage('ai_worker_agent');
    $agent = $storage->load($id);
    return $agent;
  }

  /**
   * Run workflow.
   *
   * @param \Drupal\Core\Entity\ConfigEntityInterface $agent
   *   The agent.
   * @param \Drupal\minikanban\Entity\Task $task
   *   The task.
   * @param \Drupal\minikanban_agent\PluginInterfaces\MinikanbanLlmInterface $llm
   *   The llm.
   *
   * @return \Drupal\minikanban_agent\AgentSolutions\DirectSolution|\Drupal\Core\Entity\ContentEntityInterface
   *   A direct solution or an entity.
   */
  public function runWorkflow($agent, Task $task, MinikanbanLlmInterface $llm) {
    $this->task = $task;
    $this->llm = $llm;
    // Figure out if the problem is solved already.
    $context = $this->getFullContextOfTask();
    $checkPrompt = $this->agentHelper->actionPrompts('ai_interpolator_agent', 'checkDone', [
      'Description and Comments' => $context,
    ]);
    $checkResult = $this->cleanupAndDecodeJsonResponse($this->llm->prompt($checkPrompt['prompt'], $checkPrompt));

    if (!isset($checkResult[0]['action'])) {
      throw new AgentRunnerException('Failed to check if the task is done.');
    }
    if ($checkResult[0]['action'] == 'done') {
      return new DirectSolution($checkResult[0]['extra'], 'in_review');
    }

    // Get the workflow.
    $workflow = $this->getWorkflow($agent);
    $entity = NULL;

    // If it has bundles.
    if ($workflow['bundleKey']) {
      $entity = $workflow['entityStorage']->create([
        $workflow['bundleKey'] => $workflow['bundle'],
      ]);
    }
    else {
      $entity = $workflow['entityStorage']->create([]);
    }
    // Get the default fields.
    $fields = $this->getDefaultValueFields($agent);
    foreach ($fields as $field) {
      $entity->set($field['field_name'], $field['default_value']);
    }
    // Check the required fields.
    $inputFields = $this->getInputFields($agent);
    foreach ($inputFields as $field) {
      $value = $this->askAiForValue($field);

      if (!is_null($value)) {
        $entity->set($field['field_name'], $value);
      }
      else {
        throw new AgentRunnerException('Failed to get a value for the field: ' . $field['field_name']);
      }
    }

    // Save the entity, so the AI job is run.
    $entity->save();

    return $entity;
  }

  /**
   * Gets the workflow entity for an agent.
   *
   * @param \Drupal\Core\Entity\ConfigEntityInterface $agent
   *   The agent.
   *
   * @return array
   *   An array with the entity type, bundle and storage.
   */
  public function getWorkflow($agent) {
    $workflow = $agent->get('workflow');
    $parts = explode('--', $workflow);
    $storage = $this->entityTypeManager->getStorage($parts[0]);
    $definition = $this->entityTypeManager->getDefinition($parts[0]);
    return [
      'entity_type' => $parts[0],
      'bundleKey' => $definition->getKey('bundle') ?? NULL,
      'bundle' => $parts[1],
      'entityStorage' => $storage,
    ];
  }

  /**
   * Get the workers as options.
   *
   * @param string $excludedAgent
   *   An agent to exclude.
   *
   * @return array
   *   An array of workers.
   */
  public function getWorkersAsOptions($excludedAgent = '') {
    $agents = $this->getAllAgents($excludedAgent, TRUE, FALSE);
    $options = [];
    foreach ($agents as $agent) {
      $options[$agent->id()] = $agent->label();
    }
    return $options;
  }

  /**
   * Get the input fields.
   *
   * @param \Drupal\Core\Config\Entity\EntityInterface $agent
   *   The agent.
   *
   * @return array
   *   An array of input fields.
   */
  public function getInputFields($agent) {
    $fields = $agent->get('field_connections');
    $inputFields = [];
    foreach ($fields as $field) {
      if ($field['agent_process'] == 'input') {
        $inputFields[] = $field;
      }
    }
    return array_filter($inputFields);
  }

  /**
   * Get the default value fields.
   *
   * @param \Drupal\Core\Config\Entity\EntityInterface $agent
   *   The agent.
   *
   * @return array
   *   An array of default value fields.
   */
  public function getDefaultValueFields($agent) {
    $fields = $agent->get('field_connections');
    $inputFields = [];
    foreach ($fields as $field) {
      if ($field['agent_process'] == 'default') {
        $inputFields[] = $field;
      }
    }
    return array_filter($inputFields);
  }

  /**
   * Try to get a clean value from the input.
   *
   * @param array $fieldData
   *   The field data.
   *
   * @return mixed
   *   The value.
   */
  protected function askAiForValue(array $fieldData) {
    $values = $this->getInput($fieldData);
    $cleanedValues = [];
    foreach ($values as $value) {
      switch ($value['field_type']) {
        case 'image':
        case 'file':
          $attachments = $this->task->getAttachements();
          foreach ($attachments as $attachment) {
            // TODO: Better solution.
            if (basename($attachment->getFileUri()) == basename($value['data'])) {
              $cleanedValues[] = [
                'target_id' => $attachment->id(),
              ];
            }
          }
          break;
        case 'link':
          $cleanedValues[] = [
            'uri' => $value['data'],
          ];
          break;
        default:
          $cleanedValues[] = $value['data'];
          break;
      }
    }
    return $cleanedValues;
  }

  /**
   * {@inheritDoc}
   */
  public function getInput(array $fieldData) {
    $context = $this->getFullContextOfTask(FALSE);
    $checkPrompt = $this->agentHelper->actionPrompts('ai_interpolator_agent', 'extractInput', [
      'Description and Comments' => $context,
      'Field Extract Instructions' => $fieldData['input_explanation'],
    ]);
    $checkResult = $this->cleanupAndDecodeJsonResponse($this->llm->prompt($checkPrompt['prompt'], $checkPrompt));
    if (!isset($checkResult[0]['data'])) {
      throw new AgentRunnerException('Failed to extract input from the task and comments for field ' . $fieldData['field_name'] . '.');
    }
    return $checkResult;
  }

  /**
   * Cleanup and decode json response.
   *
   * @param string $response
   *   The response.
   *
   * @return array
   *   The response.
   */
  public function cleanupAndDecodeJsonResponse($response) {
    // Remove white lines at start and finish.
    $response = trim($response);
    // Remove json notation.
    $response = str_replace(['```json', '```'], '', $response);
    // Decode the json.
    return json_decode($response, TRUE);
  }

  /**
   * Get full context of the task.
   *
   * @param bool $stripTags
   *   Strip tags.
   *
   * @return string
   *   The context.
   */
  public function getFullContextOfTask($stripTags = TRUE) {
    // Get the description and the comments.
    $context = "Task Title: " . $this->task->getTitle() . "\n";
    $context .= "Task Author: " . $this->task->getAuthorsUsername() . "\n";
    $context .= "Task Description:\n" . $this->task->getDescription();
    $context .= "\n--------------------------\n";
    if ($this->task->getCommentCount()) {
      $context .= "These are the following comments:\n";
      $context .= "--------------------------\n";
      $comments = $this->task->getComments();

      $i = 1;
      foreach ($comments as $comment) {
        $context .= "Comment order $i: \n";
        $context .= "Comment Author: " . $comment->getAuthorsUsername() . "\n";
        $context .= "Comment:\n" . $comment->getComment() . "\n\n";
        $i++;
      }
      $context .= "--------------------------\n";
    }
    return $stripTags ? strip_tags($context) : $context;
  }

}
