<?php declare(strict_types = 1);

namespace Drupal\ai_interpolator_agent\Entity;

use Drupal\ai_interpolator_agent\AiWorkerAgentInterface;
use Drupal\Core\Config\Entity\ConfigEntityBase;

/**
 * Defines the ai worker agent entity type.
 *
 * @ConfigEntityType(
 *   id = "ai_worker_agent",
 *   label = @Translation("AI Interpolator Worker Agent"),
 *   label_collection = @Translation("AI Interpolator Worker Agents"),
 *   label_singular = @Translation("AI Interpolator Worker Agent"),
 *   label_plural = @Translation("AI Interpolator Worker Agents"),
 *   label_count = @PluralTranslation(
 *     singular = "@count AI Interpolator Worker Agent",
 *     plural = "@count AI Interpolator Worker Agents",
 *   ),
 *   handlers = {
 *     "list_builder" = "Drupal\ai_interpolator_agent\AiWorkerAgentListBuilder",
 *     "form" = {
 *       "add" = "Drupal\ai_interpolator_agent\Form\AiWorkerAgentForm",
 *       "edit" = "Drupal\ai_interpolator_agent\Form\AiWorkerAgentForm",
 *       "delete" = "Drupal\Core\Entity\EntityDeleteForm",
 *     },
 *   },
 *   config_prefix = "ai_worker_agent",
 *   admin_permission = "administer ai_worker_agent",
 *   links = {
 *     "collection" = "/admin/structure/ai-worker-agent",
 *     "add-form" = "/admin/structure/ai-worker-agent/add",
 *     "edit-form" = "/admin/structure/ai-worker-agent/{ai_worker_agent}",
 *     "delete-form" = "/admin/structure/ai-worker-agent/{ai_worker_agent}/delete",
 *   },
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "description",
 *     "workflow",
 *     "retries",
 *     "time_limit",
 *     "fallback_user",
 *     "completion_rule",
 *     "remove_entity",
 *     "field_connections",
 *   },
 * )
 */
final class AiWorkerAgent extends ConfigEntityBase implements AiWorkerAgentInterface {

  /**
   * The ID.
   */
  protected string $id;

  /**
   * The label.
   */
  protected string $label;

  /**
   * The description.
   */
  protected string $description;

  /**
   * The workflow
   */
  protected string $workflow;

  /**
   * Approval type.
   */
  protected string $approval_type;

  /**
   * Retries.
   */
  protected int $retries;

  /**
   * Time limit.
   */
  protected int $time_limit;

  /**
   * Fallback user.
   */
  protected string $fallback_user;

  /**
   * Completion Status Rule
   */
  protected string $completion_rule;

  /**
   * Remove entity.
   */
  protected string $remove_entity;

  /**
   * Field connections.
   */
  protected array|null $field_connections;

  /**
   * Get the outputs.
   *
   * @return array
   *   An array of outputs.
   */
  public function getOutputs() {
    $entityBundle = explode('--', $this->get('workflow'));
    if (count($entityBundle) != 2) {
      throw new \Exception('Invalid workflow.');
    }
    $fieldData = \Drupal::service('entity_field.manager')->getFieldDefinitions($entityBundle[0], $entityBundle[1]);
    $fields = $this->get('field_connections');
    $outputs = [];
    foreach ($fields as $field) {
      if ($field['agent_process'] == 'output') {
        $outputs[$field['field_name']] = [
          'description' => $field['output_explanation'],
          'type' => $this->translateType($fieldData[$field['field_name']]->getType()),
        ];
      }
    }
    return $outputs;
  }

  /**
   * Get the input.
   *
   * @return array
   *   The inputs.
   */
  public function getInputs() {
    $entityBundle = explode('--', $this->get('workflow'));
    if (count($entityBundle) != 2) {
      throw new \Exception('Invalid workflow.');
    }
    $fieldData = \Drupal::service('entity_field.manager')->getFieldDefinitions($entityBundle[0], $entityBundle[1]);
    $fields = $this->get('field_connections');
    $inputs = [];
    foreach ($fields as $field) {
      if ($field['agent_process'] == 'input') {
        $inputs[$field['field_name']] = [
          'description' => $field['input_explanation'],
          'name' => $field['field_name'],
          'type' => $this->translateType($fieldData[$field['field_name']]->getType()),
          'default_value' => $field['default_value'] ?? '',
        ];
      }
    }
    return $inputs;
  }

  /**
   * Translate a field config type to a usable type.
   *
   * @param string $type
   *   The type.
   *
   * @return string
   *   The internal type.
   */
  public function translateType(string $type) {
    switch ($type) {
      case 'string':
      case 'string_long':
      case 'string_with_summary':
      case 'text':
      case 'text_long':
        return 'string';
      default:
        return $type;
    }
  }

}
