<?php declare(strict_types = 1);

namespace Drupal\ai_interpolator_agent\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai_interpolator_agent\Entity\AiWorkerAgent;
use Drupal\ai_interpolator_agent\EntityWorkflows;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * AI Worker Agent form.
 */
final class AiWorkerAgentForm extends EntityForm implements ContainerInjectionInterface {

  /**
   * The entity workflows.
   *
   * @var \Drupal\ai_interpolator_agent\EntityWorkflows
   */
  protected EntityWorkflows $entityWorkflows;

  /**
   * The user storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $userStorage;

  /**
   * Known outliers.
   *
   * @var array
   */
  protected array $knownOutliers = [
    'revision_timestamp',
    'revision_uid',
    'revision_log',
    'created',
    'changed',
    'sticky',
    'revision_default',
    'ai_interpolator_status',
  ];

  /**
   * Constructor fo the AiWorkerAgentForm.
   */
  public function __construct(EntityWorkflows $entityWorkflows, EntityTypeManager $entityTypeManager) {
    $this->entityWorkflows = $entityWorkflows;
    $this->userStorage = $entityTypeManager->getStorage('user');
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('ai_interpolator_agent.entity_workflows'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $formState): array {
    $form = parent::form($form, $formState);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Internal Name'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#required' => TRUE,
      '#attributes' => [
        'placeholder' => $this->t('jason_video_watcher'),
      ],
      '#description' => $this->t('This is an unique internal name for the AI Worker Agent. It has to be unique over all the agents.'),
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => [AiWorkerAgent::class, 'load'],
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->get('description'),
      '#required' => TRUE,
      '#description' => $this->t('This is a short description what this agent does. Please be very specific and fit it into 50 or less words.'),
      '#attributes' => [
        'rows' => 2,
        'placeholder' => $this->t('This agent watcher a video and describes it either in general manner or looks for what they are prompted to look at.'),
      ],
    ];

    $workflow = $formState->getValue('workflow') ?? $this->entity->get('workflow');

    $form['workflow'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Workflow'),
      '#default_value' => $workflow,
      '#required' => TRUE,
      '#description' => $this->t('This is the AI Interpolator workflow that will be used for this agent.'),
      '#ajax' => [
        'callback' => '::getWorkflow',
        'wrapper' => 'field-connections-wrapper',
        'event' => 'change',
      ],
      '#autocomplete_route_name' => 'ai_interpolator_agent.autocomplete.workflows',
    ];

    $form['retries'] = [
      '#type' => 'number',
      '#title' => $this->t('Retries'),
      '#default_value' => $this->entity->get('retries') ?? 5,
      '#description' => $this->t('The number of retries that the agent will do before giving up and assigning it to the manual user for automatic approvals. If set to 0 it continues forever, which can be costly.'),
    ];

    $form['time_limit'] = [
      '#type' => 'number',
      '#title' => $this->t('Time Limit (in seconds)'),
      '#default_value' => $this->entity->get('time_limit') ?? 600,
      '#description' => $this->t('The time the process is allowed to continue before it assigns the task back to a fallback user or the manual user.'),
    ];

    $form['fallback_user'] = [
      '#type' => 'entity_autocomplete',
      '#title' => $this->t('Fallback Manual User'),
      '#description' => $this->t('The user to be used as a fallback for doing this task, if this agent is disabled or is unable to complete the task in a satisfactory way.<br> <strong>This will be synced on uid, so set this up in settings.php if you have multiple environments.</strong>'),
      '#target_type' => 'user',
      '#default_value' => $this->entity->get('fallback_user') ? $this->userStorage->load($this->entity->get('fallback_user')) : $this->userStorage->load(1),
      '#required' => TRUE,
    ];

    $form['completion_rule'] = [
      '#type' => 'select',
      '#title' => $this->t('Completion Status Rule'),
      '#options' => [
        'in_review' => $this->t('In Review'),
        'done' => $this->t('Done'),
      ],
      '#default_value' => $this->entity->get('completion_rule'),
      '#description' => $this->t('What should happen after the rule is done? Set it depending on how much you trust the agent. <strong>Done</strong> will automatically close the task, <strong>In Review</strong> will set it to review.'),
    ];

    $form['remove_entity'] = [
      '#type' => 'select',
      '#title' => $this->t('Garbage Collect'),
      '#options' => [
        'keep' => $this->t('Always keep the entity'),
        'remove_task_delete' => $this->t('Remove on task deletion'),
        'remove' => $this->t('Remove right away'),
      ],
      '#default_value' => $this->entity->get('remove_entity'),
      '#description' => $this->t('Remove the entity from the database when the agent has successfully completed the task or a task that was closed for other reasons. <strong>Obviously do not enable this for workflows where this is the end product and end storage.</strong>'),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $this->entity->status(),
      '#description' => $this->t('If you disable an agent that has worker tasks connected to them, they will be assigned to the fallback user.'),
    ];

    $form['field_connections'] = [
      '#type' => 'details',
      '#title' => $workflow ? $this->t('Field Connection %workflow', [
        '%workflow' => $workflow
      ]) : $this->t('Choose workflow first'),
      '#attributes' => [
        'id' => 'field-connections-wrapper',
      ],
      '#tree' => TRUE,
      '#open' => $workflow,
    ];

    if ($workflow) {
      $bundleParts = explode('--', $workflow);
      $fields = $this->entityWorkflows->getFieldsForBundle($bundleParts[0], $bundleParts[1]);
      $i = 0;
      $defaultValues = $this->getDefaultValues();
      $initial = $this->getInitialValues($fields);
      foreach ($fields as $fieldName => $field) {
        // If it is a key field, we do not allow it unless its the label.
        if ((empty($field['key']) || $field['key'] === 'label') && !in_array($fieldName, $this->knownOutliers)) {
          $form['field_connections'][$i] = [
            '#type' => 'container',
          ];
          $form['field_connections'][$i]['header'] = [
            '#markup' => '<strong>' . $field['label'] . '</strong>',
          ];
          $form['field_connections'][$i]['field_name'] = [
            '#type' => 'value',
            '#value' => $field['id'],
          ];

          $form['field_connections'][$i]['agent_process'] = [
            '#type' => 'select',
            '#title' => $this->t('Agent Process'),
            '#default_value' => $defaultValues[$fieldName]['agent_process'] ?? $initial[$fieldName],
            '#options' => [
              'ignore' => $this->t('Ignore'),
              'default' => $this->t('Set Default Value'),
              'input' => $this->t('Input'),
              'output' => $this->t('Output'),
            ],
            '#description' => $this->t('This is the type of field this is to the AI agent.'),
          ];

          $form['field_connections'][$i]['input_explanation'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Input Explanation'),
            '#default_value' => $defaultValues[$fieldName]['input_explanation'] ?? '',
            '#description' => $this->t('Write a little bit of what the Manager should put as input in this field.'),
            '#attributes' => [
              'rows' => 2,
              'placeholder' => $this->t('The text that should be summarized.'),
            ],
            '#states' => [
              'visible' => [
                ':input[name="field_connections[' . $i . '][agent_process]"]' => ['value' => 'input'],
              ],
            ],
          ];

          $form['field_connections'][$i]['output_explanation'] = [
            '#type' => 'textarea',
            '#title' => $this->t('Output Explanation'),
            '#default_value' => $defaultValues[$fieldName]['output_explanation'] ?? '',
            '#description' => $this->t('Write a little bit of what the Manager can except to get as output in this field.'),
            '#attributes' => [
              'rows' => 2,
              'placeholder' => $this->t('The textual summary.'),
            ],
            '#states' => [
              'visible' => [
                ':input[name="field_connections[' . $i . '][agent_process]"]' => ['value' => 'output'],
              ],
            ],
          ];

          $form['field_connections'][$i]['default_value'] = [
            '#type' => 'textfield',
            '#title' => $this->t('Default Value'),
            '#default_value' => $defaultValues[$fieldName]['default_value'] ?? '',
            '#description' => $this->t('This is the default value that will be set if the agent does not have any input.'),
            '#states' => [
              'visible' => [
                ':input[name="field_connections[' . $i . '][agent_process]"]' => [
                  ['value' => 'default'],
                  'or',
                  ['value' => 'input'],
                ],
              ],
            ],
          ];

          $form['field_connections'][$i]['break'] = [
            '#markup' => '<br><hr><br>',
          ];
          $i++;
        }
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $formState) {
    // Validation is optional.
    $values = $formState->getValues();
    foreach ($values['field_connections'] as $key => $rules) {
      if ($rules['agent_process'] === 'default' && empty($rules['default_value'])) {
        $formState->setErrorByName('field_connections][' . $key . '][default_value', $this->t('You need to set a default value for the field.'));
      }
      if ($rules['agent_process'] === 'input' && empty($rules['input_explanation'])) {
        $formState->setErrorByName('field_connections][' . $key . '][input_explanation', $this->t('You need to set an input explanation for the field.'));
      }
      if ($rules['agent_process'] === 'output' && empty($rules['output_explanation'])) {
        $formState->setErrorByName('field_connections][' . $key . '][output_explanation', $this->t('You need to set an output explanation for the field.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $formState): int {
    $fields = $formState->getValue('field_connections');
    if (empty($fields)) {
      $this->entity->set('field_connections', []);
    }
    $result = parent::save($form, $formState);
    $message_args = ['%label' => $this->entity->label()];
    $this->messenger()->addStatus(
      match($result) {
        \SAVED_NEW => $this->t('Created new example %label.', $message_args),
        \SAVED_UPDATED => $this->t('Updated example %label.', $message_args),
      }
    );
    $formState->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

  /**
   * Get the default values if nothing is set.
   */
  private function getInitialValues(array $fields) {
    $defaults = [];
    foreach ($fields as $fieldName => $field) {
      if (!empty($field['config'])) {
        $defaults[$fieldName] = 'output';
      }
      elseif ($field['key'] == 'label') {
        $defaults[$fieldName] = 'default';
      }
      elseif ($field['required']) {
        $defaults[$fieldName] = 'input';
      }
      else {
        $defaults[$fieldName] = 'ignore';
      }
    }
    return $defaults;
  }

  /**
   * Get the actual default values.
   */
  private function getDefaultValues() {
    $connections = $this->entity->get('field_connections');
    $defaults = [];
    if (is_array($connections)) {
      foreach ($connections as $field) {
        $defaults[$field['field_name']] = $field;
      }
    }
    return $defaults;
  }

  /**
   * Ajax callback for the workflow field.
   */
  public function getWorkflow(array $form, FormStateInterface $formState): array {
    return $form['field_connections'];
  }

}
