<?php

namespace Drupal\ai_interpolator_agent\Plugin\MinikanbanAgent;

use Drupal\ai_interpolator_agent\AgentsHelper;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\minikanban_agent\AgentHelper;
use Drupal\minikanban_agent\AgentPlugin;
use Drupal\minikanban_agent\AgentSolutions\DirectSolution;
use Drupal\minikanban_agent\Annotation\MinikanbanAgent;
use Drupal\minikanban_agent\Exceptions\AgentRunnerException;
use Drupal\minikanban_agent\Exceptions\PromptException;
use Drupal\minikanban_agent\PluginInterfaces\MinikanbanAgentInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The Webform agent.
 *
 * @MinikanbanAgent(
 *   id = "ai_interpolator_agent",
 *   title = @Translation("AI Interpolator Agent"),
 * )
 */
class AiInterpolatorAgent extends AgentPlugin implements MinikanbanAgentInterface, ContainerFactoryPluginInterface {

  /**
   * The agent helper.
   *
   * @var \Drupal\ai_interpolator_agent\AgentsHelper
   */
  protected $agentHelper;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The prompt helper.
   *
   * @var \Drupal\minikanban_agent\AgentHelper
   */
  protected $promptHelper;

  /**
   * The file loader.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $fileLoader;

  /**
   * The questions.
   *
   * @var array
   */
  protected $questions = [];

  /**
   * {@inheritDoc}
   */
  public function __construct(
    AgentsHelper $agentHelper,
    AgentHelper $promptHelper,
    EntityTypeManagerInterface $entityTypeManager,
    ) {
      parent::__construct($promptHelper);
      $this->agentHelper = $agentHelper;
      $this->entityTypeManager = $entityTypeManager;
      $this->promptHelper = $promptHelper;
      $this->fileLoader = $entityTypeManager->getStorage('file');
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('ai_interpolator_agent.agents_helper'),
      $container->get('minikanban_agent.agent_helper'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getId() {
    return 'ai_interpolator_agent';
  }

  /**
   * {@inheritDoc}
   */
  public function agentsNames() {
    $agents = $this->agentHelper->getAllAgents('', TRUE, FALSE);
    $returnAgents = [];
    foreach ($agents as $agent) {
      $returnAgents[$agent->id()] = $agent->label();
    }
    return $returnAgents;
  }

  /**
   * {@inheritDoc}
   */
  public function agentsCapabilities() {
    $agents = $this->agentHelper->getAllAgents('', TRUE, FALSE);

    $returnAgents = [];
    foreach ($agents as $agent) {
      $returnAgents[$agent->id()] = [
        'name' => $agent->label(),
        'description' => $agent->get('description'),
        'outputs' => $agent->getOutputs(),
        'inputs' => $agent->getInputs(),
      ];
    }
    return $returnAgents;
  }

  /**
   * {@inheritDoc}
   */
  public function setSuggestedInputs() {
    return $this->agentHelper->getAgent($this->subAgent)->get('description');
  }

  /**
   * {@inheritDoc}
   */
  public function isAvailable() {
    return $this->agentHelper->getAgent($this->subAgent)->get('status') ?? FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function getRetries() {
    return $this->agentHelper->getAgent($this->subAgent)->get('retries') ?? 5;
  }

  /**
   * {@inheritDoc}
   */
  public function answerQuestion() {
    return "string";
  }

  /**
   * {@inheritDoc}
   */
  public function getHelp() {
    return "string";
  }

  /**
   * {@inheritDoc}
   */
  public function determineSolvability() {
    $context = $this->getFullContextOfTask($this->task);
    $worker = $this->agentHelper->getAgent($this->subAgent);
    $inputs = $worker->getInputs();
    $inputFields = '';
    foreach ($inputs as $input) {
      $inputFields .= $input['name'] . " (" . $input['type'] . ")";
      $inputFields .= $input['default_value'] ? " has default value\n" : "\n";
      $inputFields .= "Description: " . $input['description'] . "\n\n";
    }
    $finalPrompt = $this->promptHelper->actionPrompts('ai_interpolator_agent', 'solvability', [
      'Prompt (A PM ticket and comments)' => $context,
      'Input Fields' => $inputFields,
    ]);

    $data = $this->cleanupAndDecodeJsonResponse($this->llm->prompt($finalPrompt['prompt'], $finalPrompt));
    if (!isset($data[0]) || (!isset($data[0]['action']) && $data[0]['action'] === 'no_information')) {
      throw new PromptException($data[0]['value'] ?? 'No information about solvability.');
    }

    if ($data[0]['action'] === 'ask_question') {
      foreach ($data as $question) {
        if ($question['action'] === 'ask_question') {
          $this->questions[] = $question['extra'];
        }
      }
      return MinikanbanAgentInterface::JOB_SOLVABLE;
    }
    else if ($data[0]['action'] == 'solvable') {
      return MinikanbanAgentInterface::JOB_SOLVABLE;
    }
    else {
      return MinikanbanAgentInterface::JOB_NOT_SOLVABLE;
    }

  }

  /**
   * {@inheritDoc}
   */
  public function askQuestion() {
    return $this->questions;
  }

  /**
   * {@inheritDoc}
   */
  public function solve() {
    $subAgent = $this->agentHelper->getAgent($this->subAgent);
    $return = $this->agentHelper->runWorkflow($subAgent, $this->task, $this->llm);
    // Check if $return is a DirectSolution, then return otherwise its and entity.
    if ($return instanceof DirectSolution) {
      return $return;
    }
    // Get the output values from the entity.
    $outputValues = $this->getOutputValues($subAgent, $return);
    $this->setResults($outputValues);

    // Garbage collect.
    $this->garbageCollectionOrSaving($subAgent, $return);
    // Create the comment and set to review.
    if (!empty($outputValues)) {
      return new DirectSolution($this->generateResponse($outputValues), $subAgent->get('completion_rule'));
    }
    else {
      throw new AgentRunnerException('Failed to get output values.');
    }
  }

  /**
   * Gets the generated values.
   *
   * @param \Drupal\Core\Config\Entity\EntityInterface $agent
   *   The agent.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   *
   * @return array
   *   An array of output values.
   */
  public function getOutputValues($agent, $entity) {
    $fields = $agent->get('field_connections');
    $outputValues = [];

    foreach ($fields as $field) {
      if ($field['agent_process'] == 'output') {
        $fieldConfig = $entity->get($field['field_name'])->getFieldDefinition();
        $cleaned = $this->cleanValue($fieldConfig, $entity->get($field['field_name'])->getValue());
        $outputValues[$fieldConfig->getLabel()] = $cleaned;
      }
    }
    // Save the output values in a state.
    $state = $this->getState();
    $state['output_values'] = $outputValues;
    $this->setState($state);
    return $outputValues;
  }

  /**
   * Garbage collection or saving.
   *
   * @param \Drupal\ai_interpolator_agent\Entity\AiWorkerAgent $agent
   *   The agent.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity.
   */
  public function garbageCollectionOrSaving($agent, ContentEntityInterface $entity) {
    $removal = $agent->get('remove_entity');
    if ($removal == 'remove') {
      $entity->delete();
    }
    elseif ($removal == 'remove_task_delete') {
      $state = $this->getState();
      $state['garbage_entities'][] = $entity->id();
      $this->setState($state);
    }
  }

  /**
   * Clean the value.
   *
   * @param \Drupal\Core\Field\FieldConfig $fieldDefinition
   *   The field definition.
   * @param mixed $values
   *   The values.
   *
   * @return mixed
   *   The cleaned value.
   */
  public function cleanValue(FieldConfig $fieldConfig, $values) {
    if (empty($values)) {
      throw new \Exception('Something went wrong when trying to clean the values.');
    }

    $cleaned = [];
    switch ($fieldConfig->getType()) {
      case 'file':
        foreach ($values as $value) {
          /** @var \Drupal\file\Entity\File */
          $file = $this->fileLoader->load($value['target_id']);
          $cleaned[] = $this->outputFileToComment($file);
        }
        break;
      case 'image':
        foreach ($values as $value) {
          /** @var \Drupal\file\Entity\File */
          $file = $this->fileLoader->load($value['target_id']);
          $cleaned[] = $this->outputImageToComment($file);
        }
        break;
      case 'string':
      case 'string_long':
        // Add row breaks for HTML formatting of comments.
        foreach ($values as $value) {
          $cleaned[] = $this->outputStringToComment($value['value']);
        }
        break;
      default:
        foreach ($values as $value) {
          $cleaned[] = $value['value'];
        }
        break;
    }
    return $cleaned;
  }

  /**
   * Generates a response back.
   *
   * @param array $outputValues
   *   The output values.
   *
   * @return string
   *   The response.
   */
  public function generateResponse($outputValues) {
    $text = t("The AI agent has completed the task. Here are the results:<br><br>");
    foreach ($outputValues as $key => $value) {
      $text .= "<strong>$key:</strong><br>";
      foreach ($value as $part) {
        $text .= $part . '<br>';
      }
      $text .= '<br>';
    }
    return $text;
  }

}
