<?php

namespace Drupal\ai_interpolator_deepgram;

use Drupal\ai_interpolator\Annotation\AiInterpolatorFieldRule;
use Drupal\ai_interpolator\PluginInterfaces\AiInterpolatorFieldRuleInterface;
use Drupal\ai_interpolator_deepgram\Deepgram;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Utility\Token;
use Drupal\file\FileRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

class DeepgramText extends AiInterpolatorFieldRule implements AiInterpolatorFieldRuleInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   */
  public EntityTypeManagerInterface $entityManager;

  /**
   * The Deepgram requester.
   */
  public Deepgram $deepgram;

  /**
   * The File System interface.
   */
  public FileSystemInterface $fileSystem;

  /**
   * The File Repo.
   */
  public FileRepositoryInterface $fileRepo;

  /**
   * The token system to replace and generate paths.
   */
  public Token $token;

  /**
   * The current user.
   */
  public AccountProxyInterface $currentUser;

  /**
   * The logger channel.
   */
  public LoggerChannelFactoryInterface $loggerChannel;

  /**
   * Break rows.
   */
  public $breakRows = "\n";

  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityManager
   *   The entity type manager.
   * @param \Drupal\ai_interpolator_deepgram\Deepgram $deepgram
   *   The Deepgram requester.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The File system interface.
   * @param \Drupal\file\FileRepositoryInterface $fileRepo
   *   The File repo.
   * @param \Drupal\Core\Utility\Token $token
   *   The token system.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannel
   *   The logger channel interface.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entityManager,
    Deepgram $deepgram,
    FileSystemInterface $fileSystem,
    FileRepositoryInterface $fileRepo,
    Token $token,
    AccountProxyInterface $currentUser,
    LoggerChannelFactoryInterface $loggerChannel,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityManager = $entityManager;
    $this->deepgram = $deepgram;
    $this->fileSystem = $fileSystem;
    $this->fileRepo = $fileRepo;
    $this->token = $token;
    $this->currentUser = $currentUser;
    $this->loggerChannel = $loggerChannel;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('ai_interpolator_deepgram.api'),
      $container->get('file_system'),
      $container->get('file.repository'),
      $container->get('token'),
      $container->get('current_user'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function needsPrompt() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function advancedMode() {
    return FALSE;
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "";
  }

  /**
   * {@inheritDoc}
   */
  public function allowedInputs() {
    return [
      'file',
    ];
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $form['interpolator_deepgram_target_model'] = [
      '#type' => 'select',
      '#title' => 'Target Model',
      '#required' => TRUE,
      '#options' => [
        'general' => 'General',
        'meeting' => 'Meeting',
        'phonecall' => 'Phonecall',
        'voicemail' => 'Voicemail',
        'finance' => 'Finance',
        'conversationalai' => 'Conversational AI',
        'video' => 'Video',
        'whisper' => 'Whisper',
      ],
      '#description' => $this->t('The target model to use.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_deepgram_target_model', 'general'),
    ];

    $form['interpolator_deepgram_diarize'] = [
      '#type' => 'checkbox',
      '#title' => 'Diarization',
      '#description' => 'Enable speaker diarization.',
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_deepgram_diarize', FALSE),
    ];

    $form['interpolator_deepgram_timecodes'] = [
      '#type' => 'checkbox',
      '#title' => 'Timecodes',
      '#description' => 'Enable timecodes.',
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_deepgram_timecodes', FALSE),
    ];

    $form['interpolator_deepgram_multiple_fields'] = [
      '#type' => 'checkbox',
      '#title' => 'Multiple Fields',
      '#description' => 'Enable multiple fields. This stores each paragraph in a separate field.',
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_deepgram_multiple_fields', FALSE),
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $interpolatorConfig, $delta = 0) {
    $config['model'] = $interpolatorConfig['deepgram_target_model'];
    $config['diarize'] = $interpolatorConfig['deepgram_diarize'] ? 'true' : 'false';
    $config['smart_format'] = 'true';
    $values = [];
    foreach ($entity->{$interpolatorConfig['base_field']} as $entityWrapper) {
      if ($entityWrapper->entity) {
        $fileEntity = $entityWrapper->entity;
        if (in_array($fileEntity->getMimeType(), [
          'audio/mpeg',
          'audio/aac',
          'audio/wav',
        ])) {
          $return = $this->deepgram->transcribe($fileEntity, $config);

          if (!empty($return['results']['channels'][0]['alternatives'][0]['paragraphs']['paragraphs'])) {
            $paragraph = '';
            foreach ($return['results']['channels'][0]['alternatives'][0]['paragraphs']['paragraphs'] as $section) {
              foreach ($section['sentences'] as $sentence) {
                $value = '';
                if ($interpolatorConfig['deepgram_diarize']) {
                  $value .= 'Speaker ' . $section['speaker'] . ': ';
                }
                if ($interpolatorConfig['deepgram_timecodes']) {
                  $value .= '(Timecode: ' . round($sentence['start'], 3) . ' - ' . round($sentence['end'],3) . ') ';
                }
                $value .= $sentence['text'];
                // Store in one field or multiple.
                if ($interpolatorConfig['deepgram_multiple_fields']) {
                  $values[] = $value;
                } else {
                  $paragraph .= $value . $this->breakRows;
                }
              }
              if (!$interpolatorConfig['deepgram_multiple_fields']) {
                $paragraph .= $this->breakRows;
              }
            }
            if (!$interpolatorConfig['deepgram_multiple_fields']) {
              $values[] = $paragraph;
            }
          }
        }
      }
    }
    return $values;
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition) {
    // Should be a string.
    if (!is_string($value)) {
      return FALSE;
    }
    // Otherwise it is ok.
    return TRUE;
  }

}
