<?php

namespace Drupal\ai_interpolator_dreamstudio;

use Drupal\ai_interpolator_dreamstudio\Form\DreamStudioConfigForm;
use Drupal\Core\Config\ConfigFactory;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7;

/**
 * Dream Studio API creator.
 */
class DreamStudio {

  /**
   * The http client.
   */
  protected Client $client;

  /**
   * API Key.
   */
  private string $apiKey;

  /**
   * The base path.
   */
  private string $basePath = 'https://api.stability.ai/v1/';

  /**
   * Constructs a new DreamStudio object.
   *
   * @param \GuzzleHttp\Client $client
   *   Http client.
   * @param \Drupal\Core\Config\ConfigFactory $configFactory
   *   The config factory.
   */
  public function __construct(Client $client, ConfigFactory $configFactory) {
    $this->client = $client;
    $this->apiKey = $configFactory->get(DreamStudioConfigForm::CONFIG_NAME)->get('api_key') ?? '';
  }

  /**
   * Get Engines.
   *
   * @return array
   *   Engine key and name.
   */
  public function engines() {
    if (!$this->apiKey) {
      return [];
    }

    return json_decode($this->makeRequest("engines/list", [], 'GET')->getContents(), TRUE);
  }

  /**
   * Create image from text.
   *
   * @param string $prompt
   *   The prompt to use.
   * @param string $style_preset
   *   The style preset.
   * @param string $engine_id
   *   Dream Studios Engine.
   * @param int $width
   *   The width.
   * @param int $height
   *   The height.
   * @param int $steps
   *   The amount of steps.
   * @param array $options
   *   Other options to add.
   *
   * @return string
   *   Binary.
   */
  public function generateImage($prompt, $style_preset = 'photographic', $engine_id = 'stable-diffusion-xl-beta-v2-2-2', $width = 512, $height = 512, $steps = 150, $options = []) {
    if (!$this->apiKey) {
      return [];
    }
    $data = $options;
    $data['height'] = $height;
    $data['width'] = $width;
    $data['style_preset'] = $style_preset;
    $data['text_prompts'][] = [
      'text' => $prompt,
      'weight' => 1,
    ];
    $data['steps'] = $steps;
    $guzzleOptions['headers']['accept'] = 'image/png';
    $guzzleOptions['headers']['Content-Type'] = 'application/json';
    return $this->makeRequest("generation/$engine_id/text-to-image", [], 'POST', json_encode($data), $guzzleOptions)->getContents();
  }

  /**
   * Create image from text.
   *
   * @param string $uri
   *   The image binary.
   * @param string $scale
   *   How much to scale it up.
   * @param string $outputFormat
   *   The output format.
   * @param string $model
   *   The model.
   *
   * @return string
   *   Binary.
   */
  public function upscaleImage($binary, $width = 0, $height = 0, $engine = 'esrgan-v1-x2plus') {
    if (!$this->apiKey || (!$width && !$height)) {
      return [];
    }
    $extra = $width ? ['name' => 'width', 'contents' => $width] : ['name' => 'height', 'contents' => $height];
    $guzzleOptions['headers']['accept'] = 'image/png';
    $guzzleOptions['multipart'] = [
      [
        'name' => 'image',
        'contents' => $binary,
      ],
      $extra,
    ];

    return $this->makeRequest("generation/$engine/image-to-image/upscale", [], 'POST', "", $guzzleOptions)->getContents();
  }

  /**
   * Image to image augmentation.
   *
   * @param string $prompt
   *   The prompt to use.
   * @param string $binary
   *   The image binary to use.
   * @param string $style_preset
   *   The style preset.
   * @param string $engine_id
   *   Dream Studios Engine.
   * @param int $steps
   *   The amount of steps.
   * @param array $options
   *   Other options to add.
   *
   * @return string
   *   Binary.
   */
  public function augmentImage($prompt, $binary, $style_preset = 'photographic', $engine_id = 'stable-diffusion-xl-beta-v2-2-2', $steps = 150, $options = []) {
    if (!$this->apiKey) {
      return [];
    }
    $multipart = [];
    foreach ($options as $key => $value) {
      $multipart[] = [
        'name' => $key,
        'contents' => $value,
      ];
    }
    $data = $options;
    $multipart[] = [
      'name' => 'init_image',
      'contents' => $binary,
    ];
    $multipart[] = [
      'name' => 'style_preset',
      'contents' => $style_preset,
    ];
    $multipart[] = [
      'name' => 'steps',
      'contents' => $steps,
    ];
    $multipart[] = [
      'name' => 'text_prompts[0][text]',
      'contents' => $prompt,
    ];
    $multipart[] = [
      'name' => 'text_prompts[0][weight]',
      'contents' => 1,
    ];
    $data['text_prompts'][] = [
      'text' => $prompt,
      'weight' => 1,
    ];
    $guzzleOptions['multipart'] = $multipart;
    $guzzleOptions['headers']['accept'] = 'image/png';
    $result = $this->makeRequest("generation/$engine_id/image-to-image", [], 'POST', NULL, $guzzleOptions);
    return $result;
  }

  /**
   * Make DreamStudio call.
   *
   * @param string $path
   *   The path.
   * @param array $query_string
   *   The query string.
   * @param string $method
   *   The method.
   * @param string $body
   *   Data to attach if POST/PUT/PATCH.
   * @param array $options
   *   Extra headers.
   *
   * @return string|object
   *   The return response.
   */
  protected function makeRequest($path, array $query_string = [], $method = 'GET', $body = '', array $options = []) {
    // We can wait some.
    $options['connect_timeout'] = 30;
    $options['read_timeout'] = 30;
    // Don't let Guzzle die, just forward body and status.
    $options['http_errors'] = FALSE;
    // Headers.
    $options['headers']['Authorization'] = 'Bearer ' . $this->apiKey;

    if ($body) {
      $options['body'] = $body;
    }

    $new_url = $this->basePath . $path;
    $new_url .= count($query_string) ? '?' . http_build_query($query_string) : '';

    $res = $this->client->request($method, $new_url, $options);

    return $res->getBody();
  }

}
