<?php

namespace Drupal\ai_interpolator_dreamstudio\Plugin\AiInterPolatorFieldRules;

use Drupal\ai_interpolator\Annotation\AiInterpolatorFieldRule;
use Drupal\ai_interpolator\PluginBaseClasses\ImageToImage;
use Drupal\ai_interpolator\PluginInterfaces\AiInterpolatorFieldRuleInterface;
use Drupal\ai_interpolator_dreamstudio\DreamStudio;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The rules for an image outpainting field.
 *
 * @AiInterpolatorFieldRule(
 *   id = "ai_interpolator_dreamstudio_outpainting_v2",
 *   title = @Translation("DreamStudio Outpainting v2"),
 *   field_rule = "image",
 *   target = "file"
 * )
 */
class OutpaintingV2 extends ImageToImage implements AiInterpolatorFieldRuleInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'DreamStudio Outpainting v2';

  /**
   * The DreamStudio requester.
   */
  public DreamStudio $dreamStudio;

  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\ai_interpolator_dreamstudio\DreamStudio $dreamStudio
   *   The DreamStudio requester.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    DreamStudio $dreamStudio,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->dreamStudio = $dreamStudio;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai_interpolator_dreamstudio.api'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function helpText() {
    return $this->t("Generate images based on a string field using SD2. The prompt is just used to add support words to the context that has to be a plain text.");
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $cardinality = $fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $defaultGeneration = $cardinality < 0 || $cardinality > 10 ? 10 : $cardinality;
    $textAmount = $cardinality == -1 ? 'unlimited' : $cardinality;

    // Add the prompt field.
    $form['interpolator_dreamstudio_prompt'] = [
      '#type' => 'textarea',
      '#title' => 'Prompt',
      '#description' => $this->t('The prompt of how the outpainting should be done. Its optional.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_prompt', ''),
    ];
    // Allow the negative prompt to be filled from tokens.
    $this->addTokenConfigurationFormField('interpolator_dreamstudio_prompt', $form, $entity, $fieldDefinition);

    // We offer three different types of logic for the outpainting.
    $form['interpolator_dreamstudio_outpainting_choice'] = [
      '#type' => 'select',
      '#title' => 'DreamStudio Outpainting Option',
      '#options' => [
        'ratio' => $this->t('Ratio'),
        'hard_numbers' => $this->t('Hard numbers'),
        'output_size' => $this->t('Output size'),
      ],
      '#description' => $this->t('Choose the option you want to use here.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_outpainting_choice', 'hard_numbers'),
    ];

    // Ratio.
    $form['interpolator_dreamstudio_ratio'] = [
      '#type' => 'textfield',
      '#title' => 'Ratio',
      '#description' => $this->t('The ratio of the outpainting. As in 16:9. This would outpaint on the sides that are missing.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_ratio', 0),
      '#states' => [
        'visible' => [
          ':input[name="interpolator_dreamstudio_outpainting_choice"]' => ['value' => 'ratio'],
        ],
      ],
    ];

    // Hard Numbers.
    foreach ([
      'left',
      'right',
      'top',
      'bottom',
    ] as $side) {
      $form['interpolator_dreamstudio_' . $side] = [
        '#type' => 'number',
        '#title' => ucfirst($side),
        '#description' => $this->t('The amount of pixels to outpaint on the ' . $side . '.'),
        '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_' . $side, 0),
        '#states' => [
          'visible' => [
            ':input[name="interpolator_dreamstudio_outpainting_choice"]' => ['value' => 'hard_numbers'],
          ],
        ],
      ];
    }

    // Output Size.
    $form['interpolator_dreamstudio_output_size'] = [
      '#type' => 'textfield',
      '#title' => 'Output Size',
      '#description' => $this->t('The output size of the outpainting. As in 1920x1080. This would outpaint to this size.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_output_size', '1920x1080'),
      '#states' => [
        'visible' => [
          ':input[name="interpolator_dreamstudio_outpainting_choice"]' => ['value' => 'output_size'],
        ],
      ],
    ];

    $form['interpolator_image_generation_amount'] = [
      '#type' => 'number',
      '#title' => 'Generation Amount',
      '#description' => $this->t('Amount of images to generate. Generation costs money, so make sure to set this correct. You can set %amount image(s).', [
        '%amount' => $textAmount,
      ]),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_image_generation_amount', $defaultGeneration),
    ];

    $form['interpolator_image_dreamstudio_type'] = [
      '#type' => 'select',
      '#title' => 'File Type',
      '#options' => [
        'jpeg' => 'JPG',
        'png' => 'PNG',
        'webp' => 'WebP',
      ],
      '#description' => $this->t('Choose the file type you want to use here.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_image_dreamstudio_type', 'jpg'),
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function extraAdvancedFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {

    $form['interpolator_dreamstudio_option_seed'] = [
      '#type' => 'number',
      '#title' => 'DreamStudio Seed',
      '#description' => $this->t('Number of diffusion seed to run. Between 0-4294967294.'),
      '#min' => 0,
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_option_seed', 0),
      '#weight' => 24,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generateFileResponse(File $file, $interpolatorConfig, ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $options = [];
    foreach ($interpolatorConfig as $key => $value) {
      if (strpos($key, 'dreamstudio_option_') === 0 && !empty($value)) {
        $options[str_replace('dreamstudio_option_', '', $key)] = $value;
      }
    }
    $prompt = $this->getConfigValue('dreamstudio_prompt', $interpolatorConfig, $entity) ?? '';
    $left = $right = $top = $bottom = 0;
    switch ($interpolatorConfig['dreamstudio_outpainting_choice']) {
      case 'ratio':
        $ratioNumbers = explode(':', $interpolatorConfig['dreamstudio_ratio']);
        $ratio = $ratioNumbers[0] / $ratioNumbers[1];
        $size = getimagesize($file->getFileUri());
        $currentRatio = $size[0] / $size[1];
        // If it is the exact same we just return the file.
        if ($currentRatio == $ratio) {
          return file_get_contents($file->getFileUri());
        }
        // If its under we outpaint on the sides.
        if ($currentRatio < $ratio) {
          $top = 0;
          $bottom = 0;
          $left = round(($size[1] * $ratio - $size[0]) / 2, 0);
          $right = round(($size[1] * $ratio - $size[0]) / 2, 0);
        }
        // If its over we outpaint on the top and bottom.
        if ($currentRatio > $ratio) {
          $left = 0;
          $right = 0;
          $top = round(($size[0] / $ratio - $size[1]) / 2, 0);
          $bottom = round(($size[0] / $ratio - $size[1]) / 2, 0);
        }
        break;

      case 'hard_numbers':
        foreach (['left', 'right', 'top', 'bottom'] as $side) {
          ${$side} = $interpolatorConfig['dreamstudio_' . $side] ?? 0;
        }
        break;

      case 'output_size':
        $size = getimagesize($file->getFileUri());
        $wanted = explode('x', $interpolatorConfig['output_size']);
        if ($wanted[0] > $size[0])  {
          $right = ceil($wanted[0] - $size[0] / 2);
          $left = floor($wanted[0] - $size[0] / 2);
        }
        if ($wanted[1] > $size[1])  {
          $bottom = ceil($wanted[1] - $size[1] / 2);
          $top = floor($wanted[1] - $size[1] / 2);
        }
        break;
    }
    return $this->dreamStudio->outpaintImage(file_get_contents($file->getFileUri()), $left, $right, $top, $bottom, $prompt, $interpolatorConfig['image_dreamstudio_type'], $options);
  }

  /**
   * {@inheritDoc}
   */
  public function getFileName(array $args = []) {
    return 'dreamstudio.' . $args['image_dreamstudio_type'];
  }
}
