<?php

namespace Drupal\ai_interpolator_dreamstudio\Plugin\AiInterPolatorFieldRules;

use Drupal\ai_interpolator\Annotation\AiInterpolatorFieldRule;
use Drupal\ai_interpolator\PluginBaseClasses\TextToImage;
use Drupal\ai_interpolator\PluginInterfaces\AiInterpolatorFieldRuleInterface;
use Drupal\ai_interpolator_dreamstudio\DreamStudio;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The rules for an image structure field.
 *
 * @AiInterpolatorFieldRule(
 *   id = "ai_interpolator_dreamstudio_structure_v2",
 *   title = @Translation("DreamStudio Structure v2"),
 *   field_rule = "image",
 *   target = "file"
 * )
 */
class StructureV2 extends TextToImage implements AiInterpolatorFieldRuleInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'DreamStudio Structure v2';

  /**
   * The DreamStudio requester.
   */
  public DreamStudio $dreamStudio;

  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\ai_interpolator_dreamstudio\DreamStudio $dreamStudio
   *   The DreamStudio requester.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    DreamStudio $dreamStudio,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->dreamStudio = $dreamStudio;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('ai_interpolator_dreamstudio.api'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function helpText() {
    return $this->t("Generate images based on a string field and a structure image using SD2. The prompt is just used to add support words to the context that has to be a plain text.");
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $cardinality = $fieldDefinition->getFieldStorageDefinition()->getCardinality();
    $defaultGeneration = $cardinality < 0 || $cardinality > 10 ? 10 : $cardinality;
    $textAmount = $cardinality == -1 ? 'unlimited' : $cardinality;

    $imageFields = $this->getGeneralHelper()->getFieldsOfType($entity, 'image');
    $form['interpolator_dreamstudio_image'] = [
      '#type' => 'select',
      '#title' => 'Structure Image',
      '#description' => $this->t('The image structure to use together with the prompt.'),
      '#options' => $imageFields,
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_image', ''),
    ];

    // Add the prompt field.
    $form['interpolator_dreamstudio_option_negative_prompt'] = [
      '#type' => 'textarea',
      '#title' => 'Negative Prompt',
      '#description' => $this->t('The negative prompt is used to give the AI a hint of what not to do. This is a plain text.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_option_negative_prompt', ''),
    ];
    // Allow the negative prompt to be filled from tokens.
    $this->addTokenConfigurationFormField('interpolator_image_negative_prompt', $form, $entity, $fieldDefinition);

    $form['interpolator_image_generation_amount'] = [
      '#type' => 'number',
      '#title' => 'Generation Amount',
      '#description' => $this->t('Amount of images to generate. Generation costs money, so make sure to set this correct. You can set %amount image(s).', [
        '%amount' => $textAmount,
      ]),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_image_generation_amount', $defaultGeneration),
    ];

    $form['interpolator_dreamstudio_option_control_strength'] = [
      '#type' => 'number',
      '#title' => 'Control Strength',
      '#description' => $this->t('The strength of the control. Between 0-1.'),
      '#min' => 0,
      '#max' => 1,
      '#step' => 0.01,
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_option_control_strength', 0.7),
    ];

    $form['interpolator_image_dreamstudio_type'] = [
      '#type' => 'select',
      '#title' => 'File Type',
      '#options' => [
        'jpeg' => 'JPG',
        'png' => 'PNG',
        'webp' => 'WebP',
      ],
      '#description' => $this->t('Choose the file type you want to use here.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_image_dreamstudio_type', 'jpg'),
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function extraAdvancedFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $form['interpolator_dreamstudio_option_seed'] = [
      '#type' => 'number',
      '#title' => 'DreamStudio Seed',
      '#description' => $this->t('Number of diffusion seed to run. Between 0-4294967294.'),
      '#min' => 0,
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_dreamstudio_option_seed', 0),
      '#weight' => 24,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function validateConfigValues(&$form, FormStateInterface $formState) {
    // Make sure the image is selected and its enabled.
    if (!($formState->getValue('interpolator_dreamstudio_image')) && $formState->getValue('interpolator_enabled')) {
      $formState->setErrorByName('interpolator_dreamstudio_image', $this->t('You have to select a structure image.'));
    }
  }

  /**
   * {@inheritDoc}
   */
  public function generateResponse($prompt, $interpolatorConfig, ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $options = [];
    foreach ($interpolatorConfig as $key => $value) {
      if (strpos($key, 'dreamstudio_option_') === 0 && !empty($value)) {
        $options[str_replace('dreamstudio_option_', '', $key)] = $value;
      }
    }
    $imageBinary = file_get_contents($entity->get($interpolatorConfig['dreamstudio_image'])->entity->getFileUri());
    $options['negative_prompt'] = $this->getConfigValue('dreamstudio_option_negative_prompt', $interpolatorConfig, $entity);
    return $this->dreamStudio->structure($imageBinary, $prompt, $interpolatorConfig['image_dreamstudio_type'], $options);
  }

  /**
   * {@inheritDoc}
   */
  public function getFileName(array $args = []) {
    return 'dreamstudio.' . $args['image_dreamstudio_type'];
  }

}
