<?php

namespace Drupal\ai_interpolator_openai;

use Drupal\ai_interpolator\Exceptions\AiInterpolatorRequestErrorException;
use Drupal\ai_interpolator\Exceptions\AiInterpolatorResponseErrorException;
use Drupal\Core\Field\FieldDefinitionInterface;
use OpenAI\Client;
use OpenAI\Exceptions\ErrorException;

/**
 * A wrapper around OpenAI.
 */
class OpenAiRequester {

  /**
   * The OpenAI client.
   */
  protected Client $client;

  /**
   * Constructs a new CampaignListBuilder object.
   *
   * @param \OpenAI\Client $client
   *   The OpenAI client.
   */
  public function __construct(Client $client,) {
    $this->client = $client;
  }

  /**
   * Generate response.
   *
   * @param string $prompt
   *   The prompt.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $fieldDefinition
   *   The field definition interface.
   * @param array $interpolatorConfig
   *   The interpolator config.
   *
   * @return array
   *   The array of values.
   */
  public function generateResponse(string $prompt, FieldDefinitionInterface $fieldDefinition, array $interpolatorConfig) {
    try {
      $response = $this->openAiChatRequest($prompt, $interpolatorConfig);
    }
    catch (\Exception $e) {
      throw new AiInterpolatorRequestErrorException($e->getMessage());
    }

    if (isset($response['choices'][0]['message']['content'])) {
      $json = json_decode(str_replace("\n", "", $response['choices'][0]['message']['content']), TRUE);
      // Sometimes it doesn't become a valid JSON response, but many.
      if (!is_array($json)) {
        $newJson = [];
        foreach (explode("\n", $response['choices'][0]['message']['content']) as $row) {
          if ($row) {
            $parts = json_decode(str_replace("\n", "", $row), TRUE);
            if (is_array($parts)) {
              $newJson = array_merge($newJson, $parts);
            }
          }
        }
        if (!empty($newJson)) {
          $json = $newJson;
        }
      }
      if (isset($json[0]['value'])) {
        $values = [];
        foreach ($json as $val) {
          if (isset($val['value'])) {
            $values[] = $val['value'];
          }
        }
        return $values;
      }
      // Sometimes it does not return with values in GPT 3.5.
      elseif (is_array($json) && isset($json[0][0])) {
        $values = [];
        foreach ($json as $vals) {
          foreach ($vals as $val) {
            if (isset($val)) {
              $values[] = $val;
            }
          }
        }
        return $values;
      }
      elseif (isset($json['value'])) {
        return [$json['value']];
      }
      else {
        return [$response['choices'][0]['message']['content']];
      }
    }
    throw new AiInterpolatorResponseErrorException('The response did not follow the wanted structure.');
  }

  /**
   * Get all models.
   *
   * @return array
   *   The key and name of the models.
   */
  public function getModels() {
    try {
      $response = $this->client->models()->list();
    }
    catch (ErrorException $e) {
      return [];
    }

    // Only allow 3.5 or 4 based.
    $models = [];
    foreach ($response->data as $result) {
      // Currently there is no better way to search.
      // This would includ finetuned models based on these.
      if (substr($result->root, 0, 7) == 'gpt-3.5' || substr($result->root, 0, 5) == 'gpt-4' || substr($result->root, 0, 7) == 'sahara:') {
        $models[$result->id] = $result->id;
      }
    }
    return $models;
  }

  /**
   * Moderate a query.
   *
   * @param string $prompt
   *   The prompt.
   * @param array $interpolatorConfig
   *   The interpolator config.
   *
   * @return bool
   *   If the moderation is flagged or not.
   */
  public function hasFlaggedContent($prompt, array $interpolatorConfig) {
    $response = $this->client->moderations()->create([
      'input' => $prompt,
    ]);

    foreach ($response->results as $result) {
      if ($result->flagged) {
        return TRUE;
      }
    }
    return FALSE;
  }

  /**
   * Generate the OpenAI Chat request.
   *
   * @param string $prompt
   *   The prompt.
   * @param array $interpolatorConfig
   *   The interpolator config.
   *
   * @return string
   *   The response.
   */
  protected function openAiChatRequest($prompt, $interpolatorConfig) {
    if (!empty($interpolatorConfig['openai_role'])) {
      $messages[] = [
        'role' => 'system',
        'content' => trim($interpolatorConfig['openai_role']),
      ];
    }
    $messages[] = [
      'role' => 'user',
      'content' => trim($prompt),
    ];
    try {
      $response = $this->client->chat()->create([
        'model' => $interpolatorConfig['openai_model'] ?? 'gpt-3.5-turbo',
        'messages' => $messages,
      ]);
    }
    catch (ErrorException $e) {
      return $e->getMessage();
    }

    $result = $response->toArray();
    return $result;
  }

}
