<?php

namespace Drupal\ai_interpolator_openai\Plugin\AiInterPolatorFieldRules;

use Drupal\ai_interpolator\PluginInterfaces\AiInterpolatorFieldRuleInterface;
use Drupal\ai_interpolator_openai\OpenAiBase;
use Drupal\ai_interpolator_openai\OpenAiRequester;
use Drupal\Component\Utility\Bytes;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Utility\Token;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * The rules for an image field.
 *
 * @AiInterpolatorFieldRule(
 *   id = "ai_interpolator_openai_image",
 *   title = @Translation("OpenAI Image Fetcher"),
 *   field_rule = "image",
 *   target = "file"
 * )
 */
class OpenAiImageFetcher extends OpenAiBase implements AiInterpolatorFieldRuleInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritDoc}
   */
  public $title = 'OpenAI Image Fetcher';

  /**
   * The entity type manager.
   */
  public EntityTypeManagerInterface $entityManager;

  /**
   * The OpenAI requester.
   */
  public OpenAiRequester $openAi;

  /**
   * The File System interface.
   */
  public FileSystemInterface $fileSystem;

  /**
   * The token system to replace and generate paths.
   */
  public Token $token;

  /**
   * The current user.
   */
  public AccountProxyInterface $currentUser;

  /**
   * The logger channel factory.
   */
  public LoggerChannelFactoryInterface $loggerChannel;

  /**
   * Construct an image field.
   *
   * @param array $configuration
   *   Inherited configuration.
   * @param string $plugin_id
   *   Inherited plugin id.
   * @param mixed $plugin_definition
   *   Inherited plugin definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityManager
   *   The entity type manager.
   * @param \Drupal\ai_interpolator_openai\OpenAiRequester $openAi
   *   The OpenAI requester.
   * @param \Drupal\Core\File\FileSystemInterface $fileSystem
   *   The File system interface.
   * @param \Drupal\Core\Utility\Token $token
   *   The token system.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannel
   *   The logger channel factory.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entityManager,
    OpenAiRequester $openAi,
    FileSystemInterface $fileSystem,
    Token $token,
    AccountProxyInterface $currentUser,
    LoggerChannelFactoryInterface $loggerChannel,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $openAi);
    $this->entityManager = $entityManager;
    $this->openAi = $openAi;
    $this->fileSystem = $fileSystem;
    $this->token = $token;
    $this->currentUser = $currentUser;
    $this->loggerChannel = $loggerChannel;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('ai_interpolator_openai.request'),
      $container->get('file_system'),
      $container->get('token'),
      $container->get('current_user'),
      $container->get('logger.factory'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function placeholderText() {
    return "Based on the context text collect all image urls.\n\nContext:\n{{ raw_context }}";
  }

  /**
   * {@inheritDoc}
   */
  public function extraFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $form['interpolator_image_description'] = [
      '#markup' => '<strong>This will download images that are returned as full urls from the prompt if they fit validation and if they can be downloaded.</strong>',
    ];
    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function extraAdvancedFormFields(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition) {
    $form = parent::extraAdvancedFormFields($entity, $fieldDefinition);
    $form['interpolator_openai_image_title'] = [
      '#type' => 'checkbox',
      '#title' => 'Base title and alt on image tag',
      '#description' => $this->t('If the input is HTML, this will automatically fill out the values based on set values.'),
      '#default_value' => $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_openai_image_title', FALSE),
      '#weight' => 25,
    ];

    return $form;
  }

  /**
   * {@inheritDoc}
   */
  public function generate(ContentEntityInterface $entity, FieldDefinitionInterface $fieldDefinition, array $interpolatorConfig) {
    $prompts = parent::generate($entity, $fieldDefinition, $interpolatorConfig);

    $total = [];
    // Add to get functional output.
    foreach ($prompts as $i => $prompt) {
      $prompt .= "\n\nDo not include any explanations, only provide a RFC8259 compliant JSON response following this format without deviation.\n[[{\"value\": [\"uri\": \"only url\", \"title\": \"image title if exists\", \"alt\": \"image alt text if exists\"}],[{\"value\": [\"uri\": \"only url\", \"title\": \"image title if exists\", \"alt\": \"image alt text if exists\"}]]";
      try {
        $values = $this->openAi->generateResponse($prompt, $fieldDefinition, $interpolatorConfig);
        $total = array_merge_recursive($total, $values);
      }
      catch (\Exception $e) {

      }
    }
    return $total;
  }

  /**
   * {@inheritDoc}
   */
  public function verifyValue(ContentEntityInterface $entity, $value, FieldDefinitionInterface $fieldDefinition) {
    // We verify while creating, since we need to download.
    return TRUE;
  }

  /**
   * {@inheritDoc}
   */
  public function storeValues(ContentEntityInterface $entity, array $values, FieldDefinitionInterface $fieldDefinition) {
    // Initial values.
    $config = $fieldDefinition->getConfig($entity->bundle())->getSettings();
    $trashCan = [];
    $fileEntities = [];
    $successFul = 0;

    // Cleanup the values if needed.
    $values = $this->cleanUpValues($values);
    foreach ($values as $value) {
      $tmpFile = $this->fileSystem->tempnam($this->fileSystem->getTempDirectory(), 'openai_image_');
      if (empty($value['uri'])) {
        continue;
      }

      // Download image to verify.
      file_put_contents($tmpFile, file_get_contents($value['uri']));
      // Add to trash.
      $trashCan[] = $tmpFile;
      // Get base name, without potential query strings.
      $fileName = explode('?', basename($value['uri']))[0];
      // Run the validation here.
      if (!$this->validateImage($tmpFile, $fileName, $config)) {
        continue;
      }

      // Get the whole filepath.
      $filePath = $this->token->replace($config['uri_scheme'] . '://' . rtrim($config['file_directory'], '/')) . '/' . $fileName;
      $file = $this->generateFileFromString($tmpFile, $filePath);
      // If we can save, we attach it.
      if ($file) {
        $shouldCheckValues = $fieldDefinition->getConfig($entity->bundle())->getThirdPartySetting('ai_interpolator', 'interpolator_openai_image_title', FALSE);
        // Get resolution.
        $resolution = getimagesize($file->uri->value);
        // Add to the entities saved.
        $fileEntities[] = [
          'target_id' => $file->id(),
          'alt' => $this->calculateImageTags($config, $value, 'alt', $shouldCheckValues),
          'title' => $this->calculateImageTags($config, $value, 'title', $shouldCheckValues),
          'width' => $resolution[0],
          'height' => $resolution[1],
        ];

        $successFul++;
        // If we have enough images, give up.
        if ($successFul == $fieldDefinition->getFieldStorageDefinition()->getCardinality()) {
          break;
        }
      }
    }

    // Remove files.
    foreach ($trashCan as $garbageFile) {
      if (file_exists($garbageFile)) {
        unlink($garbageFile);
      }
    }

    // Then set the value.
    $entity->set($fieldDefinition->getName(), $fileEntities);
  }

  /**
   * Cleans up values because OpenAI is inconsistent.
   *
   * @param array $values
   *   The values array.
   *
   * @return array
   *   The cleaned values.
   */
  private function cleanupValues(array $values) {
    // Sometimes it comes back as a structured array in GPT 3.5.
    $newValues = [];
    if (isset($values[0]['value'])) {
      foreach ($values as $value) {
        $newValues[] = $value['value'];
      }
      $values = $newValues;
    }
    // Sometimes it comes back as a double array in GPT 3.5.
    elseif (isset($values[0][0])) {
      foreach ($values as $keys) {
        if (is_array($keys)) {
          foreach ($keys as $value) {
            $newValues[] = $value;
          }
        }
      }
      $values = $newValues;
    }
    return $values;
  }

  /**
   * Calculate tag values.
   *
   * @param array $config
   *   The field config.
   * @param array $value
   *   The value array.
   * @param string $type
   *   The type to search for.
   * @param bool $shouldCheckValues
   *   If we should check in the AI values.
   *
   * @return string
   *   The tag.
   */
  private function calculateImageTags(array $config, array $value, string $type, bool $shouldCheckValues = FALSE) {
    $new = $config['default_image'][$type] ?? '';
    if ($shouldCheckValues) {
      $new = !empty($value[$type]) ? $value[$type] : $new;
    }
    return $new;
  }

  /**
   * Validate so the image can save.
   *
   * @param string $tmpFile
   *   The location of the temporary file.
   * @param string $fileName
   *   The filename to be used.
   * @param array $config
   *   The image config.
   *
   * @return bool
   *   True if valid, otherwise false.
   */
  private function validateImage(string $tmpFile, string $fileName, array $config) {
    // Validate extension if needed.
    $extension = pathinfo($fileName, PATHINFO_EXTENSION);
    if (!empty($config['file_extensions']) && !in_array($extension, explode(' ', $config['file_extensions']))) {
      $this->loggerChannel->get('ai_interpolator')->warning('The image %fileName could not be created because it had the wrong extension. It has %extension', [
        '%fileName' => $fileName,
        '%extension' => $extension,
      ]);
      return FALSE;
    }
    // Validate file size if needed.
    if (!empty($config['max_filesize']) && filesize($tmpFile) > Bytes::toNumber($config['max_filesize'])) {
      $this->loggerChannel->get('ai_interpolator')->warning('The image %fileName was too big. It was %size bytes', [
        '%fileName' => $fileName,
        '%size' => filesize($tmpFile),
      ]);
      return FALSE;
    }
    $resolution = getimagesize($tmpFile);
    // If its a spoof file, stop.
    if (!isset($resolution[0])) {
      return FALSE;
    }
    // Validate image resolutions.
    $min = !empty($config['min_resolution']) ? explode('x', strtolower($config['min_resolution'])) : [];
    $max = !empty($config['max_resolution']) ? explode('x', strtolower($config['max_resolution'])) : [];
    if ((isset($min[0]) && $resolution[0] <= $min[0]) || (isset($min[1]) && $resolution[1] <= $min[1]) ||
    (isset($max[0]) && $resolution[1] >= $max[0]) || (isset($max[1]) && $resolution[1] >= $max[1])) {
      $this->loggerChannel->get('ai_interpolator')->warning('The resolution of image %fileName was not within bounds. The image is %x X %y', [
        '%fileName' => $fileName,
        '%x' => $resolution[0],
        '%y' => $resolution[1],
      ]);
      return FALSE;
    }
    return TRUE;
  }

  /**
   * Generate a file entity.
   *
   * @param string $source
   *   The source file.
   * @param string $dest
   *   The destination.
   *
   * @return \Drupal\file\FileInterface|false
   *   The file or false on failure.
   */
  private function generateFileFromString(string $source, string $dest) {
    // File storage.
    $fileStorage = $this->entityManager->getStorage('file');
    // Calculate path.
    $fileName = basename($dest);
    $path = substr($dest, 0, -(strlen($dest) + 1));
    // Create directory if not existsing.
    $this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY);
    $filePath = $this->fileSystem->copy($source, $dest, FileSystemInterface::EXISTS_RENAME);
    // Create file entity.
    $file = $fileStorage->create([
      'filename' => $fileName,
      'uri' => $filePath,
      'uid' => $this->currentUser->id(),
      'status' => 1,
    ]);
    if ($file->save()) {
      return $file;
    }
    return FALSE;
  }

}
