<?php

namespace Drupal\ai_provider_anthropic\Plugin\AiProvider;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Enum\AiModelCapability;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\OperationType\Chat\Tools\ToolsFunctionOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use OpenAI\Client;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'anthropic' provider.
 */
#[AiProvider(
  id: 'anthropic',
  label: new TranslatableMarkup('Anthropic'),
)]
class AnthropicProvider extends AiProviderClientBase implements
  ChatInterface {

  use ChatTrait;

  /**
   * The Anthropic Client.
   *
   * @var \OpenAI\Client|null
   */
  protected $client;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * Run moderation call, before a normal call.
   *
   * @var bool
   */
  protected bool $moderation = TRUE;

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    $models = $this->fetchAvailableModels();

    // Apply capability filtering.
    if (in_array(AiModelCapability::ChatJsonOutput, $capabilities)) {
      return array_filter($models, function ($id) {
        // Keep models that support JSON output.
        // Updated to handle various model ID formats.
        return preg_match('/claude-3\.[57]|claude-3-[57]|claude-4|claude-(opus|sonnet)-4/i', $id);
      }, ARRAY_FILTER_USE_KEY);
    }

    if ($operation_type == 'chat') {
      return $models;
    }

    return $models;
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If its not configured, it is not usable.
    if (!$this->apiKey && !$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the bundles that Anthropic supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_anthropic.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    $definition = Yaml::parseFile($this->moduleHandler->getModule('ai_provider_anthropic')->getPath() . '/definitions/api_defaults.yml');
    return $definition;
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    // If it 4.1 or higher of anything, we hide top_p.
    if (preg_match('/claude-(opus|sonnet)-4(\.[1-9]|-[1-9])/i', $model_id)) {
      unset($generalConfig['top_p']);
    }
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Set the new API key and reset the client.
    $this->apiKey = $authentication;
    $this->client = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $this->loadClient();
    // Normalize the input if needed.
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = [];
      // Add a system role if wanted.
      if ($this->chatSystemRole) {
        $chat_input[] = [
          'role' => 'system',
          'content' => $this->chatSystemRole,
        ];
      }
      /** @var \Drupal\ai\OperationType\Chat\ChatMessage $message */
      foreach ($input->getMessages() as $message) {
        $content = [
          [
            'type' => 'text',
            'text' => $message->getText(),
          ],
        ];
        if (count($message->getImages())) {
          foreach ($message->getImages() as $image) {
            $content[] = [
              'type' => 'image_url',
              'image_url' => [
                'url' => $image->getAsBase64EncodedString(),
              ],
            ];
          }
        }
        $new_message = [
          'role' => $message->getRole(),
          'content' => $content,
        ];

        if ($message->getRole() == 'tool') {
          $new_message = [
            'role' => 'tool',
            'content' => $message->getText(),
          ];
        }

        // If its a tools response.
        if ($message->getToolsId()) {
          $new_message['tool_call_id'] = $message->getToolsId();
        }

        // If we want the results from some older tools call.
        if ($message->getTools()) {
          $new_message['tool_calls'] = $message->getRenderedTools();
        }

        $chat_input[] = $new_message;
      }
    }

    $payload = [
      'model' => $model_id,
      'messages' => $chat_input,
    ] + $this->configuration;
    // If we want to add tools to the input.
    if (method_exists($input, 'getChatTools') && $input->getChatTools()) {
      $payload['tools'] = $input->getChatTools()->renderToolsArray();
      foreach ($payload['tools'] as $key => $tool) {
        $payload['tools'][$key]['function']['strict'] = FALSE;
      }
    }
    // Check for structured json schemas.
    if (method_exists($input, 'getChatStructuredJsonSchema') && $input->getChatStructuredJsonSchema()) {
      $payload['response_format'] = [
        'type' => 'json_schema',
        'json_schema' => $input->getChatStructuredJsonSchema(),
      ];
    }
    try {
      $response = $this->client->chat()->create($payload)->toArray();
      // If tools are generated.
      $tools = [];
      if (!empty($response['choices'][0]['message']['tool_calls'])) {
        foreach ($response['choices'][0]['message']['tool_calls'] as $tool) {
          $arguments = Json::decode($tool['function']['arguments']);
          $tools[] = new ToolsFunctionOutput($input->getChatTools()->getFunctionByName($tool['function']['name']), $tool['id'], $arguments);
        }
      }
      $message = new ChatMessage($response['choices'][0]['message']['role'], $response['choices'][0]['message']['content'] ?? "", []);
      if (!empty($tools)) {
        $message->setTools($tools);
      }
    }
    catch (\Exception $e) {
      throw $e;
    }

    return new ChatOutput($message, $response, []);
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    try {
      $models = $this->getConfiguredModels();
    }
    catch (\Exception $e) {
      // If we fail to get models dynamically, fall back to empty array.
      $models = [];
    }

    // Get the 4.5 models for complex tasks from the list.
    $default_complex_model = 'claude-opus-4-5-20251101';
    foreach ($models as $model_id => $model_name) {
      if (str_starts_with($model_id, 'claude-opus-4-5')) {
        // We found a 4.5 model, we can use it.
        $default_complex_model = $model_id;
        break;
      }
    }
    // Get the 4.5 sonnet model for general tasks from the list.
    $default_chat_model = 'claude-sonnet-4-5-20250929';
    foreach ($models as $model_id => $model_name) {
      if (str_starts_with($model_id, 'claude-sonnet-4-5')) {
        // We found a 4.5 sonnet model, we can use it.
        $default_chat_model = $model_id;
        break;
      }
    }

    $setup['key_config_name'] = 'api_key';
    if ($default_complex_model) {
      $setup['default_models']['chat_with_complex_json'] = $default_complex_model;
      $setup['default_models']['chat_with_tools'] = $default_complex_model;
      $setup['default_models']['chat_with_structured_response'] = $default_complex_model;
    }
    if ($default_chat_model) {
      $setup['default_models']['chat'] = $default_chat_model;
      $setup['default_models']['chat_with_image_vision'] = $default_chat_model;
    }
    return $setup;
  }

  /**
   * Enables moderation response, for all next coming responses.
   */
  public function enableModeration(): void {
    $this->moderation = TRUE;
  }

  /**
   * Disables moderation response, for all next coming responses.
   */
  public function disableModeration(): void {
    $this->moderation = FALSE;
  }

  /**
   * Gets the raw client.
   *
   * @param string $api_key
   *   If the API key should be hot swapped.
   *
   * @return \OpenAI\Client
   *   The OpenAI Client for Anthropic
   */
  public function getClient(string $api_key = ''): Client {
    if ($api_key) {
      $this->setAuthentication($api_key);
    }
    $this->loadClient();
    return $this->client;
  }

  /**
   * Loads the Anthropic Client with authentication if not initialized.
   */
  protected function loadClient(): void {
    if (!$this->client) {
      if (!$this->apiKey) {
        $this->setAuthentication($this->loadApiKey());
      }
      $host = 'https://api.anthropic.com/v1';
      $client = \OpenAI::factory()
        ->withApiKey($this->apiKey)
        ->withBaseUri($host)
        ->withHttpClient($this->httpClient);

      // If the configuration has a custom endpoint, we set it.
      if (!empty($this->getConfig()->get('host'))) {
        $client->withBaseUri($this->getConfig()->get('host'));
      }

      $this->client = $client->make();
    }
  }

  /**
   * Fetches available models from Anthropic API.
   *
   * @return array
   *   Array of models keyed by model ID with display names as values.
   */
  protected function fetchAvailableModels(): array {
    // Check cache first.
    $cache_key = 'ai_provider_anthropic:models';
    $cached = $this->cacheBackend->get($cache_key);
    if ($cached && !empty($cached->data)) {
      return $cached->data;
    }

    try {
      // Ensure we have an API key.
      $api_key = $this->apiKey ?: $this->loadApiKey();

      // Make direct HTTP request to models endpoint.
      // Note: The models endpoint requires version 2023-06-01 specifically.
      $response = $this->httpClient->request('GET', 'https://api.anthropic.com/v1/models', [
        'headers' => [
          'x-api-key' => $api_key,
          // Models endpoint requires this specific version.
          'anthropic-version' => '2023-06-01',
          'Content-Type' => 'application/json',
        ],
        'timeout' => 30,
      ]);

      $body = $response->getBody()->getContents();
      $data = Json::decode($body);

      $models = [];
      if (!empty($data['data']) && is_array($data['data'])) {
        foreach ($data['data'] as $model) {
          if (!empty($model['id']) && !empty($model['display_name'])) {
            $models[$model['id']] = $model['display_name'];
          }
        }

        // Handle pagination if needed.
        if (!empty($data['has_more']) && !empty($data['last_id'])) {
          // For now, we'll limit to first page to avoid too many requests.
          // This could be expanded in the future.
          $this->loggerFactory->get('ai_provider_anthropic')
            ->notice('Additional models available via pagination, showing first page only.');
        }
      }

      // Cache for 24 hours (configurable via settings).
      $cache_ttl = $this->getConfig()->get('models_cache_ttl') ?? 86400;
      $this->cacheBackend->set($cache_key, $models, time() + $cache_ttl);

      // Log successful fetch.
      $this->loggerFactory->get('ai_provider_anthropic')
        ->info('Successfully fetched @count models from Anthropic API', ['@count' => count($models)]);

      // Log the model IDs for debugging.
      if (count($models) > 0) {
        $this->loggerFactory->get('ai_provider_anthropic')
          ->debug('Fetched models: @models', ['@models' => implode(', ', array_keys($models))]);
      }

      return $models;
    }
    catch (\Exception $e) {
      // Log error but don't throw - gracefully fall back.
      $this->loggerFactory->get('ai_provider_anthropic')
        ->warning('Failed to fetch Anthropic models dynamically: @error', ['@error' => $e->getMessage()]);

      // Return empty array - hardcoded models will still be available.
      return [];
    }
  }

  /**
   * Clears the cached models list.
   *
   * This can be called from an admin form or drush command.
   */
  public function clearModelsCache(): void {
    $this->cacheBackend->delete('ai_provider_anthropic:models');
    $this->loggerFactory->get('ai_provider_anthropic')
      ->info('Anthropic models cache cleared.');
  }

}
