<?php

namespace Drupal\ai_provider_cohere\Form;

use Drupal\ai\AiProviderPluginManager;
use Drupal\ai_provider_cohere\CohereClient;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Psr\Http\Client\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Cohere API access.
 */
class CohereConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_cohere.settings';

  private AiProviderPluginManager $aiProviderManager;
  private KeyRepositoryInterface $keyRepository;
  private ClientInterface $httpClient;

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);;

    $instance->aiProviderManager = $container->get('ai.provider');
    $instance->keyRepository = $container->get('key.repository');
    $instance->httpClient = $container->get('http_client');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'cohere_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(
    array $form,
    FormStateInterface $form_state,
  ): array {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('Cohere API Key'),
      '#description' => $this->t(
        'The API Key. Can be found on <a href="https://dashboard.cohere.com/api-keys/">https://dashboard.cohere.com/api-keys/</a>.'
      ),
      '#default_value' => $config->get('api_key'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    // Validate the api key against model listing.
    $key = $form_state->getValue('api_key');
    $api_key = $this->keyRepository->getKey($key)->getKeyValue();
    if (!$api_key) {
      $form_state->setErrorByName(
        'api_key',
        $this->t('The API Key is invalid.')
      );

      return;
    }
    $client = CohereClient::fromClientAndApiKey($this->httpClient, $api_key);
    try {
      $client->listModels();
    }
    catch (\Exception $e) {
      $form_state->setErrorByName(
        'api_key',
        $this->t('The API Key is not working.')
      );
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(
    array &$form,
    FormStateInterface $form_state,
  ): void {
    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->save();

    $this->aiProviderManager->defaultIfNone('rerank', 'cohere', 'rerank-multilingual-v3.0');

    parent::submitForm($form, $form_state);
  }

}
