<?php

namespace Drupal\ai_provider_litellm\Plugin\AiProvider;

use Drupal\ai_provider_litellm\LiteLLM\LiteLlmAiClient;
use Drupal\ai_provider_openai\Plugin\AiProvider\OpenAiProvider;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;

/**
 * Plugin implementation of the 'LiteLLM Proxy' provider.
 */
#[AiProvider(
  id: 'litellm',
  label: new TranslatableMarkup('LiteLLM Proxy'),
)]
class LiteLlmAiProvider extends OpenAiProvider {

  /**
   * The LiteLLM API client.
   *
   * @var \Drupal\ai_provider_litellm\LiteLLM\LiteLlmAiClient
   */
  protected LiteLlmAiClient $liteLlmClient;

  /**
   * {@inheritdoc}
   */
  protected function loadClient(): void {
    parent::loadClient();
    $config = $this->getConfig();
    $this->liteLlmClient = new LiteLlmAiClient(
      $this->httpClient,
      $this->keyRepository,
      $config->get('host'),
      $config->get('api_key'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_litellm.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getModels(string $operation_type, $capabilities): array {
    $models = [];
    foreach ($this->liteLlmClient->models() as $model) {
      switch ($operation_type) {
        case 'text_to_image':
          if ($model->supportsImageOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'text_to_speech':
          if ($model->supportsAudioOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'audio_to_audio':
          if ($model->supportsAudioInput && $model->supportsAudioOutput) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'moderation':
          if ($model->supportsModeration) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'embeddings':
          if ($model->supportsEmbeddings) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'chat':
          if ($model->supportsChat) {
            $models[$model->name] = $model->name;
          }
          break;

        case 'image_and_audio_to_video':
          if ($model->supportsImageAndAudioToVideo) {
            $models[$model->name] = $model->name;
          }
          break;

        default:
          break;
      }
    }
    return $models;
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    // Don't set up any default models.
    return [
      'key_config_name' => 'api_key',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function postSetup(): void {
    // Prevent the OpenAI rate limit check.
  }

}
