<?php

namespace Drupal\ai_provider_vllm\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\RequestException;
use Drupal\Core\Url;

/**
 * Configure vLLM Provider settings for this site.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The key repository.
   *
   * @var \Drupal\key\KeyRepositoryInterface
   */
  protected $keyRepository;

  /**
   * The HTTP client.
   *
   * @var \GuzzleHttp\ClientInterface
   */
  protected $httpClient;

  /**
   * Constructs a SettingsForm object.
   *
   * @param \Drupal\key\KeyRepositoryInterface $key_repository
   *   The key repository.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client.
   */
  public function __construct(KeyRepositoryInterface $key_repository, ClientInterface $http_client) {
    $this->keyRepository = $key_repository;
    $this->httpClient = $http_client;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('key.repository'),
      $container->get('http_client')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'ai_provider_vllm_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['ai_provider_vllm.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('ai_provider_vllm.settings');

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('OpenAI API Key'),
      '#description' => $this->t('The API Key.'),
      '#default_value' => $config->get('api_key'),
    ];

    $form['base_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('vLLM Base URL'),
      '#description' => $this->t('Enter the base URL of your vLLM API (e.g., http://localhost:8000/v1).'),
      '#default_value' => $config->get('base_url') ?? 'http://localhost:8000/v1',
      '#required' => TRUE,
    ];

    $form['model'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Model Name'),
      '#description' => $this->t('Enter the name of the model to use (e.g., "mistralai/Mistral-7B-Instruct-v0.2"). If unsure, check your vLLM logs.'),
      '#default_value' => $config->get('model') ?? '',
      '#required' => TRUE,
    ];

    /* Currently only chat and embeddings are supported */
    $model_types = [
      'chat' => 'chat',
      'embeddings' => 'embeddings',
      // 'moderation',
      // 'text_to_image',
      // 'text_to_speech',
      // 'speech_to_text',
    ];

    $form['model_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Model Type'),
      '#description' => $this->t('Type of model'),
      '#default_value' => $config->get('model_type') ?? '',
      '#options' => $model_types,
      '#required' => TRUE,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('ai_provider_vllm.settings')
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('base_url', $form_state->getValue('base_url'))
      ->set('model', $form_state->getValue('model'))
      ->set('model_type', $form_state->getValue('model_type'))
      ->save();

    parent::submitForm($form, $form_state);
  }
}
