<?php

namespace Drupal\ai_provider_webt\Plugin\AiProvider;

use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\OperationType\TranslateText\TranslateTextInput;
use Drupal\ai\OperationType\TranslateText\TranslateTextInterface;
use Drupal\ai\OperationType\TranslateText\TranslateTextOutput;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\webt\TranslationHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the WEB-T provider.
 */
#[AiProvider(
  id: 'webt',
  label: new TranslatableMarkup('WEB-T'),
)]
class WebTProvider extends AiProviderClientBase implements
  ContainerFactoryPluginInterface,
  TranslateTextInterface {

  /**
   * The WEB-T service.
   *
   * @var Drupal\webt\translation_engines\etranslation\EtranslationService
   */
  protected $client;

  /**
   * Dependency Injection for the Provider.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->client = $container->get('webt.translation_service.etranslation');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_webt.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    // For this provider, we only have one model.
    return ['default' => 'Default'];
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'translate_text',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    return Yaml::parseFile($this->moduleHandler->getModule('ai_provider_webt')->getPath() . '/definitions/api_defaults.yml');
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function translateText(TranslateTextInput $input, string $model_id, array $options = []): TranslateTextOutput {
    $text = $input->getText();
    $sourceLanguage = $input->getSourceLanguage();
    $targetLanguage = $input->getTargetLanguage();

    try {
      // Exctract only the parts to translate.
      $strings_to_translate = TranslationHelper::extractTranslatableHtmlValues($text);

      // Request for translation.
      $translation_response = $this->client->translate($sourceLanguage, $targetLanguage, $strings_to_translate);
    }
    catch (\Exception $e) {
      return new TranslateTextOutput('', $text, $e->getMessage());
    }

    // Build the translated values.
    $translation = TranslationHelper::mergeTranslatedHtmlValues($text, $strings_to_translate, $translation_response->translations, TRUE);

    return new TranslateTextOutput($translation, $translation, []);

  }

}
