<?php

namespace Drupal\ai_provider_yandex\Form;

use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\ai\AiProviderPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure YandexGPT API access.
 */
class YandexConfigForm extends ConfigFormBase {

  /**
   * Config settings.
   */
  const CONFIG_NAME = 'ai_provider_yandex.settings';

  /**
   * The AI provider manager.
   *
   * @var \Drupal\ai\AiProviderPluginManager
   */
  protected $aiProviderManager;

  /**
   * Module Handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * Constructs a new YandexConfigForm object.
   */
  final public function __construct(AiProviderPluginManager $ai_provider_manager, ModuleHandlerInterface $module_handler) {
    $this->aiProviderManager = $ai_provider_manager;
    $this->moduleHandler = $module_handler;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new static(
      $container->get('ai.provider'),
      $container->get('module_handler')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'yandexgpt_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      static::CONFIG_NAME,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config(static::CONFIG_NAME);

    $form['api_key'] = [
      '#type' => 'key_select',
      '#title' => $this->t('YandexGPT API Key'),
      '#description' => $this->t('The API Key. See documentation <a href="https://yandex.cloud/ru/docs/foundation-models/api-ref/authentication#service-account_1">https://yandex.cloud/ru/docs/foundation-models/api-ref/authentication#service-account_1</a>.'),
      '#default_value' => $config->get('api_key'),
    ];

    $form['catalog_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('YandexGPT Catalog Identifier'),
      '#description' => $this->t('The Catalog Identifier. See documentation <a href="https://yandex.cloud/ru/docs/resource-manager/operations/folder/get-id">https://yandex.cloud/ru/docs/resource-manager/operations/folder/get-id</a>.'),
      '#default_value' => $config->get('catalog_id'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Retrieve the configuration.
    $this->config(static::CONFIG_NAME)
      ->set('api_key', $form_state->getValue('api_key'))
      ->set('catalog_id', $form_state->getValue('catalog_id'))
      ->save();

    // Set some defaults.
    $this->aiProviderManager->defaultIfNone('chat', 'yandex', 'yandexgpt-4-pro');

    parent::submitForm($form, $form_state);
  }

}
