<?php

namespace Drupal\ai_provider_yandex\Plugin\AiProvider;

use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\ai\Attribute\AiProvider;
use Drupal\ai\Base\AiProviderClientBase;
use Drupal\ai\Exception\AiResponseErrorException;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatInterface;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\ai\OperationType\Chat\ChatOutput;
use Drupal\ai\Traits\OperationType\ChatTrait;
use Symfony\Component\Yaml\Yaml;

/**
 * Plugin implementation of the 'yandex' provider.
 */
#[AiProvider(
  id: 'yandex',
  label: new TranslatableMarkup('Yandex'),
)]
class YandexProvider extends AiProviderClientBase implements ChatInterface {

  use ChatTrait;

  /**
   * API Key.
   *
   * @var string
   */
  protected string $apiKey = '';

  /**
   * {@inheritdoc}
   */
  public function getConfiguredModels(?string $operation_type = NULL, array $capabilities = []): array {
    if ($operation_type === 'chat') {
      return [
        'yandexgpt-4-pro' => 'YandexGPT 4 Pro',
        'yandexgpt-5-pro' => 'YandexGPT 5 Pro',
        'yandexgpt-4-lite' => 'YandexGPT 4 Lite',
      ];
    }
    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function isUsable(?string $operation_type = NULL, array $capabilities = []): bool {
    // If it is not configured, it is not usable.
    if (!$this->apiKey && !$this->getConfig()->get('api_key')) {
      return FALSE;
    }
    // If its one of the bundles that YandexGPT supports its usable.
    if ($operation_type) {
      return in_array($operation_type, $this->getSupportedOperationTypes());
    }
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function getSupportedOperationTypes(): array {
    return [
      'chat',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfig(): ImmutableConfig {
    return $this->configFactory->get('ai_provider_yandex.settings');
  }

  /**
   * {@inheritdoc}
   */
  public function getApiDefinition(): array {
    // Load the configuration.
    return Yaml::parseFile($this->moduleHandler->getModule('ai_provider_yandex')->getPath() . '/definitions/api_defaults.yml');
  }

  /**
   * {@inheritdoc}
   */
  public function getModelSettings(string $model_id, array $generalConfig = []): array {
    return $generalConfig;
  }

  /**
   * {@inheritdoc}
   */
  public function setAuthentication(mixed $authentication): void {
    // Set the new API key and reset the client.
    $this->apiKey = $authentication;
  }

  /**
   * {@inheritdoc}
   */
  public function chat(array|string|ChatInput $input, string $model_id, array $tags = []): ChatOutput {
    $catalog_id = $this->getConfig()->get('catalog_id');
    $this->setAuthentication($this->loadApiKey());
    $api_key = $this->apiKey;

    // Normalize the input if needed.
    $chat_input = $input;
    if ($input instanceof ChatInput) {
      $chat_input = [];
      // Add a system role if wanted.
      if ($this->chatSystemRole) {
        $chat_input[] = [
          'role' => 'system',
          'text' => $this->chatSystemRole,
        ];
      }
      /** @var \Drupal\ai\OperationType\Chat\ChatMessage $message */
      foreach ($input->getMessages() as $message) {
        $content = [
          [
            'type' => 'text',
            'text' => $message->getText(),
          ],
        ];
        if (count($message->getImages())) {
          foreach ($message->getImages() as $image) {
            $content[] = [
              'type' => 'image_url',
              'image_url' => [
                'url' => $image->getAsBase64EncodedString(),
              ],
            ];
          }
        }
        $chat_input[] = [
          'role' => $message->getRole(),
          'text' => $content[0]['text'],
        ];
      }
    }

    switch ($model_id) {
      case 'yandexgpt-4-pro':
        $model_uri = 'yandexgpt/latest';
        break;

      case 'yandexgpt-5-pro':
        $model_uri = 'yandexgpt/rc';
        break;

      case 'yandexgpt-4-lite':
        $model_uri = 'yandexgpt-lite/latest';
        break;

      default:
        $model_uri = 'yandexgpt/latest';
    }

    $payload = [
      'modelUri' => "gpt://$catalog_id/$model_uri",
      'completionOptions' => [
        'stream' => FALSE,
        'temperature' => 0.6,
        'maxTokens' => '2000',
        'reasoningOptions' => [
          'mode' => 'DISABLED',
        ],
      ],
      'messages' => $chat_input,
    ];

    try {
      $response = $this->httpClient->post('https://llm.api.cloud.yandex.net/foundationModels/v1/completion', [
        'headers' => [
          'Content-Type' => 'application/json',
          'Authorization' => "Api-Key $api_key",
        ],
        'json' => $payload,
      ]);

      if (!$response->getBody()) {
        throw new AiResponseErrorException('Invalid response from YandexGPT');
      }

      $response_body = (string) $response->getBody();
      $response_body = json_decode($response_body, TRUE);
      $role = $response_body['result']['alternatives'][0]['message']['role'];
      $text = $response_body['result']['alternatives'][0]['message']['text'];
      $message = new ChatMessage($role, $text);
    }
    catch (\Exception $e) {
      throw $e;
    }

    return new ChatOutput($message, $response_body, []);
  }

  /**
   * {@inheritdoc}
   */
  public function getSetupData(): array {
    return [
      'key_config_name' => 'api_key',
      'default_models' => [
        'chat' => 'yandexgpt-4-pro',
      ],
    ];
  }

}
