<?php

namespace Drupal\ai_refactor\Commands;

use Drush\Commands\DrushCommands;
use PhpParser\Error;
use PhpParser\Node;
use PhpParser\NodeTraverser;
use PhpParser\NodeVisitorAbstract;
use PhpParser\ParserFactory;
use PhpParser\PrettyPrinter\Standard;
use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;

/**
 * Drush commands for analyzing and refactoring PHP methods.
 */
class MethodAnalyzerCommands extends DrushCommands
{
  /**
   * Analyzes PHP methods in a specified file for potential refactoring.
   *
   * @param string $filePath
   *   The path to the PHP file to analyze.
   *
   * @command ai_refactor:analyze
   * @aliases ma:analyze
   */
  public function analyze(string $filePath): void
  {
    try {
      $ast = $this->parseFile($filePath);
      $this->processMethods($ast, $filePath);
    } catch (\Exception $e) {
      $this->io()->writeln("[error] {$e->getMessage()}");
    }
  }

  /**
   * Parses a PHP file and returns its Abstract Syntax Tree (AST).
   *
   * @param string $filePath
   *   The path to the PHP file to parse.
   *
   * @return array
   *   The AST of the parsed file.
   *
   * @throws \Exception
   *   Throws an exception if a parsing error occurs.
   */
  private function parseFile(string $filePath): array
  {
    $content = $this->getFileContent($filePath);
    $parser = (new ParserFactory())->createForNewestSupportedVersion();

    try {
      return $parser->parse($content);
    } catch (Error $e) {
      throw new \Exception("Parsing error: {$e->getMessage()}");
    }
  }

  /**
   * Processes the methods found in the given AST for analysis.
   *
   * @param array $ast
   *   The Abstract Syntax Tree of the PHP file.
   * @param string $filePath
   *   The path to the analyzed PHP file.
   */
  private function processMethods(array $ast, string $filePath): void
  {
    $log = $this->loadLog();
    $methodCollector = new class extends NodeVisitorAbstract {
      public array $methods = [];

      public function enterNode(Node $node)
      {
        if ($node instanceof Node\Stmt\ClassMethod) {
          $visibility = $node->isPrivate() ? 'private' : ($node->isProtected() ? 'protected' : 'public');
          $methodCode = (new Standard())->prettyPrint([$node]);
          $this->methods[] = [
            'name' => $node->name->toString(),
            'visibility' => $visibility,
            'code' => $methodCode,
          ];
        }
      }
    };

    $traverser = new NodeTraverser();
    $traverser->addVisitor($methodCollector);
    $traverser->traverse($ast);

    foreach ($methodCollector->methods as $method) {
      if (isset($log[$filePath]) && in_array($method['name'], $log[$filePath])) {
        $this->io()->writeln("[info] Skipping already processed method '{$method['name']}' in file '$filePath'.");
        continue;
      }

      $this->refactorMethod($ast, $method['name'], $filePath, $method['code']);

      // Only log if the user chose "no" to avoid future prompts.
      if ($this->shouldLogAsProcessed($method['name'], $filePath)) {
        $log[$filePath][] = $method['name'];
        $this->saveLog($log);
      }
    }
  }

  /**
   * Sends the given PHP method code to an AI service for refactoring and updates the file.
   *
   * @param string $methodCode
   *   The original PHP method code to be refactored.
   * @param string $filePath
   *   The path to the original PHP file.
   * @param \PhpParser\Node\Stmt\ClassMethod $methodNode
   *   The AST node representing the method.
   */
  private function refactorMethod(array &$ast, string $methodName, string $filePath, string $originalCode = ''): void
  {
    while (true) {
      $refactoredCode = $this->generateRefactoredAst($ast, $methodName);
      $this->displayCodeComparison($originalCode, $refactoredCode);

      $choice = $this->confirmRefactor($methodName);

      if ($choice === 'yes') {
        $this->updateFileContent($filePath, $ast);
        break;
      } elseif ($choice === 'skip') {
        $this->io()->writeln("[info] Skipping method '{$methodName}' refactoring.");
        break;
      } elseif ($choice === 'no') {
        $this->io()->writeln("[info] Refactoring canceled for method '{$methodName}'.");
        $this->markMethodAsProcessed($methodName, $filePath);
        break;
      } elseif ($choice === 'try_again') {
        $this->io()->writeln("[info] Retrying refactoring for method '{$methodName}'...");
      }
    }
  }

  /**
   * Asks the user to confirm whether the suggested refactoring should be applied.
   *
   * @param string $original
   *   The original method code.
   * @param string $refactored
   *   The suggested refactored method code.
   *
   * @return bool
   *   TRUE if the user confirms the refactoring, FALSE otherwise.
   */
  private function confirmRefactor(string $methodName): string
  {
    $choices = [
      'yes' => 'Yes',
      'no' => 'No',
      'skip' => 'Skip',
      'try_again' => 'Try Again'
    ];
    $choice = $this->io()->choice("Replace original method '{$methodName}' with the refactored version?", array_keys($choices), 'no');

    return $choice;
  }

  /**
   * Generates a modified AST with the refactored method code.
   *
   * @param array $ast
   *   The original Abstract Syntax Tree of the file.
   * @param \PhpParser\Node\Stmt\ClassMethod $originalMethod
   *   The AST node representing the original method.
   * @param string $refactoredCode
   *   The refactored method code to insert.
   *
   * @return array
   *   The updated AST with the refactored method in place.
   */
  private function generateRefactoredAst(array &$ast, string $methodName): string
  {
    $traverser = new NodeTraverser();
    $refactoredCode = '';

    $logFunction = function ($message) {
      $this->io()->writeln($message);
    };

    $traverser->addVisitor(new class ($methodName, $this, $refactoredCode, $logFunction) extends NodeVisitorAbstract {
      private $logFunction;

      public function __construct(
        private string $methodName,
        private MethodAnalyzerCommands $drushCommands,
        private string &$refactoredCode,
        callable $logFunction
      ) {
        $this->logFunction = $logFunction;
      }

      public function enterNode(Node $node)
      {
        if ($node instanceof Node\Stmt\ClassMethod && $node->name->toString() === $this->methodName) {
          $methodCode = (new Standard())->prettyPrint([$node]);
          $newBody = $this->drushCommands->generateRefactoredCode($methodCode);

          if ($newBody) {
            $tempClassCode = "<?php class TempClass { " . $newBody . " }";
            $parser = (new ParserFactory())->createForNewestSupportedVersion();

            try {
              $parsedClass = $parser->parse($tempClassCode);
              if ($parsedClass && $parsedClass[0] instanceof Node\Stmt\Class_) {
                $tempMethod = $parsedClass[0]->getMethods()[0] ?? null;
                if ($tempMethod) {
                  $node->stmts = $tempMethod->stmts;
                  $this->refactoredCode = (new Standard())->prettyPrint([$node]);
                }
              }
            } catch (Error $e) {
              ($this->logFunction)("[error] Parsing error in refactored code: {$e->getMessage()}");
            }
          }
        }
      }
    });

    $traverser->traverse($ast);

    return $refactoredCode;
  }

  /**
   * Displays a side-by-side comparison of the original and refactored code.
   *
   * @param string $original
   *   The original method code.
   * @param string $refactored
   *   The refactored method code.
   */
  private function displayCodeComparison(string $originalCode, string $refactoredCode): void
  {
    $this->io()->writeln("[info] Original Code:\n" . $originalCode);
    $this->io()->writeln("\n===\n");
    $this->io()->writeln("[info] Refactored Code:\n" . $refactoredCode);
  }

  /**
 * Updates the file content with the new refactored code.
 *
 * @param string $filePath
 *   The path to the PHP file to update.
 * @param array $refactoredAst
 *   The updated Abstract Syntax Tree containing the refactored method.
 */
  private function updateFileContent(string $filePath, array $updatedAst): void
  {
    file_put_contents($filePath, (new Standard())->prettyPrintFile($updatedAst));
    $this->io()->writeln("[success] Method updated in $filePath.");
  }

  /**
   * Sends a request to the AI service to generate refactored code for the given method.
   *
   * @param string $methodCode
   *   The original PHP method code to be refactored.
   *
   * @return string
   *   The refactored version of the method code.
   *
   * @throws \Exception
   *   Throws an exception if the AI service fails or returns invalid data.
   */
  public function generateRefactoredCode(string $methodCode): string
  {
    try {
      $messages = new ChatInput([
        new ChatMessage('user', 'Update the following method to PHP 8 and Drupal 10: ' . $methodCode),
      ]);

      $provider_plugin = \Drupal::service('ai.provider')->createInstance('openai');
      $response = $provider_plugin->chat($messages, 'gpt-3.5-turbo')->getNormalized();

      return $response->getText() ?? '';
    } catch (\Exception $e) {
      $this->io()->writeln("[error] OpenAI error: {$e->getMessage()}");
      return '';
    }
  }

  /**
   * Reads the content of a file from a given path.
   *
   * @param string $filePath
   *   The path to the file.
   *
   * @return string
   *   The content of the file.
   *
   * @throws \Exception
   *   Throws an exception if the file cannot be read.
   */
  private function getFileContent(string $relativeFilePath): string
  {
    $absolutePath = DRUPAL_ROOT . '/' . ltrim($relativeFilePath, '/');
    if (!is_readable($absolutePath)) {
      throw new \Exception("File not found or unreadable: $absolutePath");
    }

    return file_get_contents($absolutePath) ?: throw new \Exception("Failed to read file: $absolutePath");
  }

  /**
   * Loads the analysis log for tracking processed methods.
   *
   * @return array
   *   The existing log data.
   */
  private function loadLog(): array
  {
    $logFile = DRUPAL_ROOT . '/ai_refactor_log.json';
    if (file_exists($logFile)) {
      $content = file_get_contents($logFile);
      return json_decode($content, true) ?: [];
    }
    return [];
  }

  /**
   * Saves the analysis log to persistent storage.
   *
   * @param array $log
   *   The log data to be saved.
   */
  private function saveLog(array $log): void
  {
    $logFile = DRUPAL_ROOT . '/ai_refactor_log.json';
    file_put_contents($logFile, json_encode($log, JSON_PRETTY_PRINT));
  }

  /**
   * Determines whether a method should be logged as processed to avoid future prompts.
   *
   * @param string $methodName
   *   The name of the method.
   * @param string $filePath
   *   The path to the PHP file containing the method.
   *
   * @return bool
   *   TRUE if the method should be marked as processed in the log, FALSE otherwise.
   */
  private function shouldLogAsProcessed(string $methodName, string $filePath): bool
  {
    // Only log if the user chose "no" to avoid being prompted again
    return in_array($methodName, $this->loadLog()[$filePath] ?? []);
  }

  /**
   * Marks a method as processed in the log to prevent repeated refactoring prompts.
   *
   * @param string $methodName
   *   The name of the method.
   * @param string $filePath
   *   The path to the PHP file containing the method.
   */
  private function markMethodAsProcessed(string $methodName, string $filePath): void
  {
    $log = $this->loadLog();
    $log[$filePath][] = $methodName;
    $this->saveLog($log);
  }
}
