<?php

namespace Drupal\ai_related_content\Plugin\views\filter;

use Drupal\ai_related_content\Plugin\views\argument\AIRelatedContentNodeArgument;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\node\NodeInterface;
use Drupal\search_api\Plugin\views\filter\SearchApiFulltext;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Filters content based on relevance to the node in the current request.
 *
 * @ingroup views_filter_handlers
 *
 * @ViewsFilter("ai_related_content_node_filter")
 */
class AIRelatedContentNodeFilter extends SearchApiFulltext implements ContainerFactoryPluginInterface {

  /**
   * Constructs a the AI Related Content Views Filter.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entityDisplayRepository
   *   The entity display repository.
   * @param \Drupal\Core\Routing\RouteMatchInterface $routeMatch
   *   The current route.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected RequestStack $requestStack,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EntityDisplayRepositoryInterface $entityDisplayRepository,
    protected RouteMatchInterface $routeMatch,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   *
   * This filter requires the 'node' entity type.
   */
  public function getEntityType() {
    return 'node';
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('current_route_match'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['view_mode'] = ['default' => 'teaser'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // No user facing search, so not applicable.
    $hide_options = [
      'operator',
      'expose_button',
      'min_length',
      'value',
      'parse_mode',
      'parse_mode_phrase_description',
      'parse_mode_terms_description',
      'parse_mode_direct_description',
    ];
    foreach ($hide_options as $hide_option) {
      if (isset($form[$hide_option])) {
        $form[$hide_option]['#access'] = FALSE;
      }
    }

    // Allow the site builder to select the View Mode. This is very important
    // to get good results as well as avoid potentially high LLM costs.
    $view_modes = $this->entityDisplayRepository->getViewModeOptions('node');
    $form['view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Node View Mode for Source Content'),
      '#options' => $view_modes,
      '#description' => $this->t('The view mode used to render the current node. It is strongly recommended to curate this to only render actual text content to avoid the vector database search finding related content based on text not key to the Node itself (e.g. such as content from this View potentially).'),
    ];
    if (array_key_exists($this->options['view_mode'], $view_modes)) {
      $form['view_mode']['#default_value'] = $this->options['view_mode'];
    }
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    if ($this->routeMatch->getRouteName() === 'entity.view.preview_form') {
      $this->addDebugHelp();
    }

    // Bail if unable to load node.
    $query = $this->getQuery();
    $current_node = $this->getNodeForRelatedContent();
    if (!$current_node) {
      $query->getSearchApiQuery()->abort();
      return;
    }

    // Bail if unable to get content from the node.
    $content = $this->getTextFromNode($current_node);
    if (empty($content)) {
      $query->getSearchApiQuery()->abort();
      return;
    }

    // Exclude the current content item.
    $query->addCondition('nid', $current_node->id(), '!=');

    // Set the current content to be the query and run the query.
    $this->value = $content;
    parent::query();
  }

  /**
   * Add warning messages when the site builder has misconfigured the View.
   */
  protected function addDebugHelp(): void {

    // Ensure that the node and its body are found.
    $current_node = $this->getNodeForRelatedContent();
    if ($current_node) {
      $content = $this->getTextFromNode($current_node);
      if (empty($content)) {
        $this->messenger()->addWarning($this->t('The Node to use to find related contents was found; however, when attempting to render it in the selected View Mode (@view_mode) no content was found and therefore no related content could be found.', [
          '@view_mode' => $this->options['view_mode'],
        ]));
      }
    }
    else {
      $this->messenger()->addWarning($this->t('When previewing Views is unaware of which Node ID you want related content for. Provide a Node ID like "123" in the "Preview with contextual filters" input.'));
    }

    if (!$this->getRelatedContentArgument()) {
      $this->messenger()->addWarning($this->t('The contextual filter for AI Related Content appears to be missing. Please ensure the "Fulltext search from current node" contextual filter is added. Choose to "Hide the view" when there are no results.'));
    }

    if (!array_key_exists('nid', $this->getIndex()->getFields())) {
      $this->messenger()->addWarning($this->t('Your Search Index must have the Node ID indexed to exclude the current node from its own related content. The machine name for the field expected is "nid". In your search index fields page, add the Node ID and set it as a "Filterable Attribute".'));
    }
  }

  /**
   * Get the node to retrieve related content for.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The node or null.
   */
  protected function getNodeForRelatedContent(): NodeInterface|null {

    // If the argument is still attached to the View as needed.
    if ($argument = $this->getRelatedContentArgument()) {
      $node = $argument->getNode();
      if ($node instanceof NodeInterface) {
        return $node;
      }
    }
    return NULL;
  }

  /**
   * Get the AI Related Content argument belonging to this module.
   *
   * @return \Drupal\ai_related_content\Plugin\views\argument\AIRelatedContentNodeArgument|null
   *   The related content argument if found.
   */
  protected function getRelatedContentArgument(): AIRelatedContentNodeArgument|null {
    if (is_array($this->view->argument)) {
      foreach ($this->view->argument as $argument) {
        if ($argument instanceof AIRelatedContentNodeArgument) {
          return $argument;
        }
      }
    }
    return NULL;
  }

  /**
   * Get a text representation of the source to find related content for.
   *
   * @param \Drupal\node\NodeInterface $node
   *   The node to get related content for.
   *
   * @return string
   *   The content.
   */
  protected function getTextFromNode(NodeInterface $node): string {

    // Render the node in the selected View Mode if it still exists.
    $view_mode = $this->options['view_mode'];
    $view_modes = $this->entityDisplayRepository->getViewModeOptions('node');
    if (!empty($view_mode) && array_key_exists($this->options['view_mode'], $view_modes)) {

      // Build the rendered content.
      $build = $this->entityTypeManager->getViewBuilder('node')->view($node, $view_mode);
      if (empty($build)) {
        return '';
      }
      $rendered_content = $this->getRenderer()->renderRoot($build);

      // Convert to markdown if available.
      if (class_exists('League\CommonMark\CommonMarkConverter')) {
        // Ignore the non-use statement loading since this dependency may
        // not exist.
        // @codingStandardsIgnoreLine
        $converter = new \League\CommonMark\CommonMarkConverter([
          'html_input' => 'strip',
          'allow_unsafe_links' => FALSE,
        ]);
        $text_content = $converter->convert($rendered_content);
        return trim($text_content);
      }
      else {
        // Fallback to plain text.
        $text_content = strip_tags($rendered_content);

        // Strip extra new lines.
        $text_content = trim(preg_replace("/\n\n+/s", "\n", $text_content));
        return trim($text_content);
      }
    }
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheMaxAge() {
    // If there is no content, bypass cache as we are bailing immediately and
    // we do not want no relate content to be cached.
    $current_node = $this->getNodeForRelatedContent();
    if (!$current_node || empty($this->getTextFromNode($current_node))) {
      return 0;
    }
    return parent::getCacheMaxAge();
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    if (!empty($this->options['exposed'])) {
      return $this->t('exposed');
    }
    $view_modes = $this->entityDisplayRepository->getViewModeOptions('node');
    $selected_view_mode = $this->options['view_mode'];
    $view_mode_label = $view_modes[$selected_view_mode] ?? $selected_view_mode;

    return $this->t('View mode: @view_mode', [
      '@view_mode' => $view_mode_label,
    ]);
  }

}
