<?php

namespace Drupal\Tests\ai_related_content\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Contains AI Search UI setup functional tests.
 *
 * @group ai_search_functional
 */
class AiRelatedContentSetupTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'ai',
    'ai_search',
    'test_ai_related_content_provider_mysql',
    'test_ai_related_content_vdb_provider_mysql',
    'node',
    'taxonomy',
    'user',
    'system',
    'field_ui',
    'views_ui',
    'ai_related_content',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with permission to bypass access content.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    if ($this->profile != 'standard') {
      $this->drupalCreateContentType([
        'type' => 'article',
        'name' => 'Article',
      ]);
    }

    $this->adminUser = $this->drupalCreateUser([
      'access administration pages',
      'administer content types',
      'access content overview',
      'administer nodes',
      'administer node fields',
      'bypass node access',
      'administer ai',
      'administer ai providers',
      'administer search_api',
      'administer views',
    ]);

    $this->setupServerAndIndex();
  }

  /**
   * Set up server and index.
   *
   * This both tests the setup and provides a default setup for other tests.
   */
  public function setupServerAndIndex(): void {
    $this->drupalLogin($this->adminUser);

    // Set the embedding default provider as the test MySQL one.
    $this->drupalGet('admin/config/ai/settings');
    $this->submitForm([
      'operation__embeddings' => 'test_mysql_related_content_provider',
    ], 'Save configuration');
    $this->submitForm([
      'operation__embeddings' => 'test_mysql_related_content_provider',
      'model__embeddings' => 'mysql',
    ], 'Save configuration');

    // Set up Search API Server.
    $this->drupalGet('admin/config/search/search-api/add-server');
    $this->submitForm([
      'name' => 'Test MySQL AI Vector Database',
      'id' => 'test_mysql_related_content_vdb',
      'backend' => 'search_api_ai_search',
      'status' => TRUE,
    ], 'Save');
    $this->submitForm([
      'backend_config[embeddings_engine]' => 'test_mysql_related_content_provider__mysql',
      'backend_config[database]' => 'test_mysql_related_content_vdb',
      'backend_config[embeddings_engine_configuration][dimensions]' => 384,
      'backend_config[embeddings_engine_configuration][set_dimensions]' => TRUE,
    ], 'Save');
    $this->submitForm([
      'backend_config[database_settings][database_name]' => 'test_mysql_related_content_database',
      'backend_config[database_settings][collection]' => 'test_mysql_related_content_collection',
    ], 'Save');

    // Set up index.
    $this->drupalGet('admin/config/search/search-api/add-index');
    $this->submitForm([
      'name' => 'Test MySQL VDB Index',
      'id' => 'test_mysql_related_content_vdb_index',
      'datasources[entity:node]' => TRUE,
      'server' => 'test_mysql_related_content_vdb',
      'options[cron_limit]' => 5,
    ], 'Save and add fields');
    $this->submitForm([], 'Save and add fields');

    // Add fields.
    $page = $this->getSession()->getPage();
    // Rendered html.
    $page->pressButton('edit-4');
    $this->submitForm([
      'view_mode[entity:node][:default]' => 'full',
    ], 'Save');
    // Title.
    $this->drupalGet('admin/config/search/search-api/index/test_mysql_related_content_vdb_index/fields/add/nojs');
    $page->pressButton('edit-23');
    // Done.
    $page->clickLink('edit-done');

    // Selecting indexing options on fields page.
    $this->submitForm([
      'fields[rendered_item][indexing_option]' => 'main_content',
      'fields[title][indexing_option]' => 'contextual_content',
    ], 'Save changes');

    // Check indexing options have been configured.
    $this->drupalGet('admin/config/search/search-api/index/test_mysql_related_content_vdb_index');
    $this->assertSession()->pageTextContains('Indexing options have been configured.');
  }

  /**
   * Install View.
   */
  public function testAiRelatedContentSetup() {

    // No access when not logged in.
    $this->drupalLogout();
    $this->drupalGet('/admin/config/ai/ai-related-content-setup');
    $this->assertSession()->statusCodeEquals(403);

    // Access denied when logged in without permission.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/ai/ai-related-content-setup');
    $this->assertSession()->statusCodeEquals(403);

    // Grant permission.
    $aiManagerRole = $this->drupalCreateRole([
      'administer ai related content',
    ], 'ai_manager');
    $this->adminUser->addRole($aiManagerRole)->save();
    $this->drupalLogout();
    $this->drupalLogin($this->adminUser);

    // Access granted.
    $this->drupalGet('/admin/config/ai/ai-related-content-setup');
    $this->assertSession()->statusCodeEquals(200);
    $this->submitForm([
      'search_api_index' => 'test_mysql_related_content_vdb_index',
    ], 'Create View');

    // Check view is created with our filter.
    $this->drupalGet('/admin/structure/views/view/ai_related_content');

    // Filter exists.
    $this->assertSession()->pageTextContains('Search: AI Related Content from node (View mode: Teaser)');

    // Argument exists.
    $this->drupalGet('admin/structure/views/nojs/handler/ai_related_content/default_block/argument/search_api_fulltext_1_ai_related_content');
    $this->assertSession()->statusCodeEquals(200);

    // @todo extend tests to actually check related content as well. This is
    // more difficult as we need the vector database in place and the FFI
    // stuff is quite overkill for here. Maybe we wait until there is something
    // that does not need extension, or we mock the results.
  }

}
