<?php

namespace Drupal\ai_related_content;

use Drupal\ai_search\Backend\AiSearchBackendPluginBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\search_api\IndexInterface;

/**
 * AI Related Content Helper.
 */
class AiRelatedContentManager {

  /**
   * Constructs the AI Related Content Helper.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The AI Related Content logger.
   */
  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    protected LoggerChannelInterface $logger,
  ) {
  }

  /**
   * Finds Search API indexes compatible with AI Related Content.
   *
   * @return array
   *   An array of compatible indexes suitable for a form options list
   *   [index_id => index_label].
   */
  public function findCompatibleIndexes(): array {
    $options = [];
    $indexes = $this->entityTypeManager->getStorage('search_api_index')->loadMultiple();

    /** @var \Drupal\search_api\IndexInterface $index */
    foreach ($indexes as $index_id => $index) {
      // Ensure the index is valid and enabled before checking the backend.
      if (!$index instanceof IndexInterface || !$index->status()) {
        continue;
      }
      try {
        $server = $index->getServerInstance();
        if ($server && $server->status() && $server->hasValidBackend()) {
          $backend = $server->getBackend();
          // Check if the backend plugin extends the required base class.
          if ($backend instanceof AiSearchBackendPluginBase) {
            $options[$index_id] = $index->label() . ' (' . $index_id . ')';
          }
        }
      }
      catch (\Exception $e) {
        // Log error if server or backend retrieval fails for an index.
        $this->logger->warning('Could not check compatibility for Search API index @index_id: @message', [
          '@index_id' => $index_id,
          '@message' => $e->getMessage(),
        ]);
      }
    }
    return $options;
  }

}
