<?php

namespace Drupal\ai_related_content\Plugin\views\argument;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\node\NodeInterface;
use Drupal\search_api\Plugin\views\argument\SearchApiStandard;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Argument handler for AI Related Content.
 *
 * @ingroup views_argument_handlers
 *
 * @ViewsArgument("ai_related_content_node_argument")
 */
class AIRelatedContentNodeArgument extends SearchApiStandard implements ContainerFactoryPluginInterface {

  /**
   * The loaded Node entity based on the argument.
   *
   * @var \Drupal\node\NodeInterface|null
   */
  protected $node = NULL;

  /**
   * Constructs a NodeIdArgument object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * Provides the node title as the argument title.
   */
  public function getTitle() {
    $node = $this->getNode();
    if ($node instanceof NodeInterface) {
      return $node->label();
    }
    return parent::getTitle();
  }

  /**
   * Get the node to find AI Related Content for.
   *
   * @return \Drupal\node\NodeInterface|null
   *   The loaded node entity, or NULL if the argument is invalid or node does
   *   not exist.
   */
  public function getNode(): ?NodeInterface {
    if (!isset($this->node)) {
      $this->node = FALSE;
      $nid = $this->getValue();
      if (is_numeric($nid) && $nid > 0) {
        $node = $this->entityTypeManager->getStorage('node')->load($nid);
        if ($node instanceof NodeInterface) {
          $this->node = $node;
        }
      }
    }
    return $this->node ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function query($group_by = FALSE) {}

}
