<?php

namespace Drupal\ai_story_teller\Drush\Commands;

use Drupal\ai\OperationType\Chat\ChatInput;
use Drupal\ai\OperationType\Chat\ChatMessage;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Utility\Token;
use Drush\Attributes as CLI;
use Drush\Commands\DrushCommands;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A Drush commandfile.
 */
final class AiStoryTellerCommands extends DrushCommands {

  /**
   * Constructs an AiStoryTellerCommands object.
   */
  public function __construct(
    private readonly Token $token,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct();
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('token'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * Remove all scraped content.
   */
  #[CLI\Command(name: 'ai_reporter:remove-all-content', aliases: ['air:empty'])]
  #[CLI\Option(name: 'profile', description: 'Profile machine name')]
  #[CLI\Option(name: 'source', description: 'Source machine name')]
  #[CLI\Usage(name: 'ai_reporters:remove-all-content news cnn', description: 'Remove all content from the news profile and cnn source')]
  public function removeAllContent($options = [
    'profile' => NULL,
    'source' => NULL,
  ]
  ): void {
    //TODO: Add filtering for profile and source options. Not it just deletes all content.
    //TODO: Move this command from global group to its own.

    $storage = $this->entityTypeManager->getStorage('ai_reporter_content');
    $entity_ids = $storage->getQuery()->accessCheck(FALSE)->execute();
    $entities = $storage->loadMultiple($entity_ids);

    if (!empty($entities)) {
      $storage->delete($entities);
      $this->logger()->success(dt('Deleted all AI Reporter content entities.'));
    }
    else {
      $this->logger()
        ->warning(dt('No AI Reporter content entities found to delete.'));
    }
  }

  /**
   * Generate AI report.
   */
  #[CLI\Command(name: 'ai_reporter:generate', aliases: ['air:gen'])]
  #[CLI\Option(name: 'profile', description: 'Profile machine name')]
  #[CLI\Usage(name: 'ai_reporter:generate news', description: 'Generate AI Report for profile "news"')]
  public function generateReport($options = ['profile' => NULL]): void {
    $profileId = $options['profile'];
    /** @var \Drupal\ai_story_teller\Entity\Profile $profile */
    $profile = $this->entityTypeManager->getStorage('profile')
      ->load($profileId);
    if ($profile) {
      $system_prompt = $profile->get('prompt_template');
      $report_storage_config = $profile->get('reports_storage');
      if (is_string($report_storage_config)) {
        $report_storage_config = json_decode($report_storage_config, TRUE);
      }
      else {
        $this->logger()
          ->error(dt('Report storage config is not set in profile.'));
      }

      $provider_settings = $profile->getProviderSettings();
      if (!empty($provider_settings)) {
        $provider_settings->model_settings = json_decode($provider_settings->model_settings, TRUE);
      }
      else {
        $this->logger()
          ->error(dt('Provider settings config is not set in profile.'));
      }
    }
    else {
      $this->logger()->error(dt('Profile not found.'));
    }

    $storage = $this->entityTypeManager->getStorage('ai_reporter_content');
    $current_date = date('Y-m-d');
    $query = $storage->getQuery()
      ->condition('created', strtotime($current_date . ' 00:00:00'), '>=')
      ->condition('created', strtotime($current_date . ' 23:59:59'), '<=')
      ->accessCheck(FALSE);
    $entity_ids = $query->execute();
    $entities = $storage->loadMultiple($entity_ids);

    if (!empty($entities)) {
      // prepare user prompt
      $json = [];
      /** @var \Drupal\ai_story_teller\Entity\AiRContent $entity */
      foreach ($entities as $entity) {
        // human readable date of created
        $human_date = date('Y-m-d H:i:s', $entity->get('created')->value);
        $json['articles'][] = [
          'title' => $entity->getLabel(),
          'full_text' => $entity->get('body')->value,
          'published_on' => $human_date,
          'url' => $entity->get('url')->value,
          'source' => $entity->get('source')->value,
        ];
      }
      $user_prompt = "Below is the JSON data containing the titles, full texts, and publication dates and times of the news articles. JSON Input:\n"
        . json_encode($json, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

      $input = new ChatInput([
        new ChatMessage("user", $user_prompt),
      ]);

      /* @var \Drupal\ai\Base\AiProviderClientBase $ai_provider */
      $ai_provider = \Drupal::service('ai.provider')
        ->createInstance($provider_settings->provider);

      $ai_provider->setConfiguration($provider_settings->model_settings);
      $ai_provider->setChatSystemRole($system_prompt);

      $ai_response = $ai_provider->chat($input, $provider_settings->model, ["ai_story_teller"])
        ->getNormalized();

      // publish report
      $this->entityTypeManager->getStorage('node')->create([
        'type' => $report_storage_config['content_type'],
        $report_storage_config['title_field'] => 'AI Report for ' . $current_date,
        $report_storage_config['body_field'] => [
          'value' => $ai_response->getText(),
          'format' => 'full_html',
        ],
      ])->save();
    }
    else {
      $this->logger()->warning(dt('No content found for toady.'));
    }
  }

}
